<?php

class XenGallery_Model_Category extends XenForo_Model
{
	public function getCategoryById($categoryId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xengallery_category
			WHERE xengallery_category_id = ?
		', $categoryId);
	}

	public function getCategoriesByIds(array $categoryIds)
	{
		if (!$categoryIds)
		{
			return array();
		}

		return $this->fetchAllKeyed('
			SELECT *
			FROM xengallery_category
			WHERE xengallery_category_id IN (' . $this->_getDb()->quote($categoryIds) . ')
		', 'xengallery_category_id');
	}

	public function getAllCategories()
	{
		return $this->fetchAllKeyed('
			SELECT *
			FROM xengallery_category
			ORDER BY display_order
		', 'xengallery_category_id');
	}

	public function getCategoryCount()
	{
		return $this->_getDb()->query('
			SELECT COUNT(*)
			FROM xengallery_category
		');
	}
	
	public function getChildCategoriesForParent($parentCategoryId)
	{
		return $this->_getDb()->fetchCol('
			SELECT xengallery_category_id
			FROM xengallery_category
			WHERE parent_category_id = ?
		', $parentCategoryId);
	}
	
	/**
	 * Group category entries by their parent.
	 * Code based on Admin Navigation Categories by XenForo Ltd.
	 *
	 * @param array $categories List of category entries to group
	 *
	 * @return array [parent category id][category id] => info
	 */
	public function groupCategoriesByParent(array $categories)
	{
		$output = array(
			'0' => array()
		);		
		foreach ($categories AS $category)
		{
			$output[$category['parent_category_id']][$category['xengallery_category_id']] = $category;
		}

		return $output;
	}

	/**
	 * Get the categories list in the correct display order. This can be processed
	 * linearly with depth markers to visually represent the tree.
	 * Code based on Admin Navigation Categories by XenForo Ltd.
	 *
	 * @param array|null $categories Category entries; if null, grabbed automatically
	 * @param integer $root Root node to traverse from
	 * @param integer $depth Depth to start at
	 *
	 * @return array [categories id] => info, with depth key set
	 */
	public function getCategoryStructure(array $categories = null, $root = 0, $depth = 0)
	{
		if (!is_array($categories))
		{
			$categories = $this->groupCategoriesByParent($this->getAllCategories());
		}

		if (!isset($categories[$root]))
		{
			return array();
		}

		$output = array();
		foreach ($categories[$root] AS $category)
		{
			$category['depth'] = $depth;
			$output[$category['xengallery_category_id']] = $category;

			$output += $this->getCategoryStructure($categories, $category['xengallery_category_id'], $depth + 1);
		}

		return $output;
	}
	
	public function prepareCategories(array $categories, array $viewingUser = null)
	{	
		$this->standardizeViewingUserReference($viewingUser);
		$userModel = $this->getModelFromCache('XenForo_Model_User');
		
		foreach ($categories AS &$category)
		{
			$category['canAddMedia'] = false;
			
			$userGroups = unserialize($category['upload_user_groups']);
			if (in_array('-1', $userGroups))
			{
				$category['canAddMedia'] = true;
			}
			
			if ($userModel->isMemberOfUserGroup($viewingUser, $userGroups))
			{
				$category['canAddMedia'] = true;
			}
		}
		
		return $categories;
	}
	
	public function getCategoryBreadcrumb(array $category, $includeSelf = true)
	{
		$breadcrumbs = array();

		if (!isset($category['categoryBreadcrumb']))
		{
			$category['categoryBreadcrumb'] = unserialize($category['category_breadcrumb']);
		}

		foreach ($category['categoryBreadcrumb'] AS $catId => $breadcrumb)
		{
			$breadcrumbs[$catId] = array(
				'href' => XenForo_Link::buildPublicLink('full:xenmediagallery/categories', $breadcrumb),
				'value' => $breadcrumb['category_title']
			);
		}

		if ($includeSelf)
		{
			$breadcrumbs[$category['xengallery_category_id']] = array(
				'href' => XenForo_Link::buildPublicLink('full:xenmediagallery/categories', $category),
				'value' => $category['category_title']
			);
		}

		return $breadcrumbs;
	}	
	
	
	public function rebuildCategoryStructure()
	{
		$grouped = $this->groupCategoriesByParent($this->getAllCategories());

		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);

		$changes = $this->_getStructureChanges($grouped);
		foreach ($changes AS $categoryId => $changes)
		{
			$db->update('xengallery_category', $changes, 'xengallery_category_id = ' . $db->quote($categoryId));
		}

		XenForo_Db::commit($db);

		return $changes;
	}

	protected function _getStructureChanges(array $grouped, $parentId = 0, $depth = 0,
		$startPosition = 1, &$nextPosition = 0, array $breadcrumb = array()
	)
	{
		$nextPosition = $startPosition;

		if (!isset($grouped[$parentId]))
		{
			return array();
		}

		$changes = array();
		$serializedBreadcrumb = serialize($breadcrumb);

		foreach ($grouped[$parentId] AS $categoryId => $category)
		{
			$nextPosition++;

			$thisBreadcrumb = $breadcrumb + array(
				$categoryId => array(
					'xengallery_category_id' => $categoryId,
					'category_title' => $category['category_title'],
					'parent_category_id' => $category['parent_category_id'],
					'depth' => $category['depth']
				)
			);

			$changes += $this->_getStructureChanges(
				$grouped, $categoryId, $depth + 1, $nextPosition, $nextPosition, $thisBreadcrumb
			);

			$catChanges = array();
			if ($category['depth'] != $depth)
			{
				$catChanges['depth'] = $depth;
			}
			if ($category['category_breadcrumb'] != $serializedBreadcrumb)
			{
				$catChanges['category_breadcrumb'] = $serializedBreadcrumb;
			}

			if ($catChanges)
			{
				$changes[$categoryId] = $catChanges;
			}

			$nextPosition++;
		}

		return $changes;
	}	
}