<?php

class XenGallery_DataWriter_Rating extends XenForo_DataWriter
{
	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xengallery_rating' => array(
				'rating_id'  => array('type' => self::TYPE_UINT, 'autoIncrement' => true),
				'media_id'         => array('type' => self::TYPE_UINT, 'required' => true),
				'user_id'             => array('type' => self::TYPE_UINT, 'required' => true),
				'username' => array('type' => self::TYPE_STRING, 'required' => true, 'maxLength' => 50,
					'requiredError' => 'please_enter_valid_username'
				),				
				'rating'              => array('type' => self::TYPE_UINT, 'required' => true, 'min' => 1, 'max' => 5),
				'rating_date'         => array('type' => self::TYPE_UINT, 'required' => true, 'default' => XenForo_Application::$time)
			)
		);
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!is_array($data))
		{
			$ratingId = $data;
		}
		
		if ($ratingId)
		{
			$rating = $this->_getRatingModel()->getRatingById($ratingId);
		}
		else
		{
			$rating = $this->_getRatingModel()->getRatingByMediaAndUserId($data['media_id'], $data['user_id']);
		}
		
		return array('xengallery_rating' => $rating);
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'media_rating_id = ' . $this->_db->quote($this->getExisting('media_rating_id'));
	}

	protected function _preSave()
	{
		if (!$this->get('user_id') || !$this->get('media_id'))
		{
			throw new XenForo_Exception(new XenForo_Phrase('xengallery_an_unexpected_error_occurred'));
		}

		if ($this->isChanged('user_id') || $this->isChanged('media_id'))
		{
			$existing = $this->_getRatingModel()->getRatingByMediaAndUserId(
				$this->get('media_id'), $this->get('user_id')
			);
			if ($existing)
			{
				throw new XenForo_Exception('xengallery_an_unexpected_error_occurred');
			}
		}
	}

	/**
	 * Post-save handling.
	 */
	protected function _postSave()
	{
		$this->_updateMedia($this->get('rating'));
		
		if ($this->isInsert())
		{
			$mediaId = $this->get('media_id');
			
			$media = $this->_getMediaModel()->getMediaById($this->get('media_id'), array(
				'join' => XenGallery_Model_Media::FETCH_USER | XenGallery_Model_Media::FETCH_USER_OPTION
			));

			if ($media && XenForo_Model_Alert::userReceivesAlert($media, 'xengallery_rating', 'insert'))
			{	
				$ratingUser = array(
					'user_id' => $this->get('user_id'),
					'username' => $this->get('username')			
				);			
				
				XenForo_Model_Alert::alert(
					$media['user_id'],
					$ratingUser['user_id'],
					$ratingUser['username'],
					'xengallery_rating',
					$this->get('rating_id'),
					'insert'
				);			
			}
						
			$this->_getNewsFeedModel()->publish(
				$this->get('user_id'),
				$this->get('username'),
				'xengallery_rating',
				$this->get('rating_id'),
				'insert'
			);			
		}
	}

	/**
	 * Post-delete handling.
	 */
	protected function _postDelete()
	{
		$this->_updateMedia($this->getExisting('rating'), true);
		
		$comment = $this->_getCommentModel()->getCommentByRatingId($this->getExisting('media_rating_id'));
		
		if ($comment)
		{
			$commentDw = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');
			$commentDw->setExistingData($comment['comment_id']);
			$commentDw->set('rating_id', 0);
			$commentDw->save();			
		}
	}

	/**
	 * Update the media table to reflect the new rating
	 *
	 * @param integer $rating
	 */
	protected function _updateMedia($rating)
	{
		$mediaDw = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
		$mediaDw->setExistingData($this->get('media_id'));

		$mediaDw->set('rating_sum', $mediaDw->get('rating_sum') + $rating - $this->getExisting('rating'));
		
		$mediaDw->updateRating();		

		$mediaDw->save();
	}

	/**
	* @return XenGallery_Model_Rating
	*/
	protected function _getRatingModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Rating');
	}

	/**
	 * @return XenGallery_Model_Media
	 */
	protected function _getMediaModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Media');
	}

	/**
	 * @return XenGallery_Model_Comment
	 */
	protected function _getCommentModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Comment');
	}	
}