<?php

class XenGallery_ControllerHelper_Media extends XenForo_ControllerHelper_Abstract
{
	/**
	 * Checks that a category is valid and viewable, before returning the category's info.
	 *
	 * @param integer|null $id Category ID
	 * @param array $fetchOptions Extra data to fetch with the category
	 *
	 * @return array Forum info
	 */
	public function assertCategoryValidAndViewable($categoryId)
	{
		$category = $this->_getCategoryOrError($categoryId);

		return $category;
	}
	
	public function assertMediaValidAndViewable($mediaId, $fetchOptions = array())
	{
		$mediaModel = $this->_getMediaModel();
		$attachmentModel = $this->_getAttachmentModel();
		
		if (!$fetchOptions)
		{
			$fetchOptions = $this->_getMediaFetchOptions();
		}
		
		$media = $mediaModel->getMediaById($mediaId, $fetchOptions);
		
		if (!$media)
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException('');
		}
		
		if ($media['media_type'] == 'image_upload')
		{
			$media = $mediaModel->prepareMedia($media);
		}
		
		if (!$mediaModel->canViewDeletedMedia($error) && $media['media_state'] == 'deleted')
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);			
		}
		
		if (!$mediaModel->canViewUnapprovedMedia($media, $error) && $media['media_state'] == 'moderated')
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
		
		return $media;
	}
	
	public function assertCanAddMedia()
	{
		if (!$this->_getMediaModel()->canAddMedia($error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}
	
	public function assertCanEditMedia($media)
	{
		if (!$this->_getMediaModel()->canEditMedia($media, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}
	
	public function assertCanEditCaption($media)
	{
		if (!$this->_getMediaModel()->canEditCaption($media, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}	
	
	public function assertCanRateMedia($media)
	{
		if (!$this->_getMediaModel()->canRateMedia($media, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}
	
	public function assertCanDeleteMedia($media, $type = 'soft')
	{
		if (!$this->_getMediaModel()->canDeleteMedia($media, $type, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}
	
	public function assertCanViewComments()
	{
		if (!$this->_getCommentModel()->canViewComments($error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}
	}
	
	public function assertCanAddComment()
	{
		if (!$this->_getCommentModel()->canAddComment($error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}
	}	
	
	public function assertCommentValidAndViewable($commentId)
	{
		$commentModel = $this->_getCommentModel();
		
		$comment = $commentModel->getCommentById($commentId, $this->_getCommentFetchOptions());
		
		if (!$comment)
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException('');
		}
		
		if (!$commentModel->canViewDeletedComment($error) && $comment['comment_state'] == 'deleted')
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);			
		}
		
		return $comment;
	}	
	
	public function assertCanEditComment($comment)
	{
		if (!$this->_getCommentModel()->canEditComment($comment, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}	
	
	public function assertCanDeleteComment($comment, $type = 'soft')
	{
		if (!$this->_getCommentModel()->canDeleteComment($comment, $type, $error))
		{
			throw $this->_controller->getErrorOrNoPermissionResponseException($error);
		}		
	}	
		
	/**
	 * Gets the specified record or errors.
	 *
	 * @param string $categoryId
	 *
	 * @return array
	 */
	protected function _getCategoryOrError($categoryId)
	{
		$category = $this->_getCategoryModel()->getCategoryById($categoryId);
		if (!$category)
		{
			throw $this->_controller->responseException($this->_controller->responseError(new XenForo_Phrase('requested_category_not_found'), 404));
		}

		return $category;
	}
	
	protected function _getMediaFetchOptions()
	{
		$mediaFetchOptions = array(
			'join' => XenGallery_Model_Media::FETCH_USER | XenGallery_Model_Media::FETCH_CATEGORY | XenGallery_Model_Media::FETCH_ATTACHMENT
		);
		
		if (XenForo_Visitor::getInstance()->hasPermission('xengallery', 'viewDeleted'))
		{
			$mediaFetchOptions['join'] |= XenGallery_Model_Media::FETCH_DELETION_LOG;
		}

		return $mediaFetchOptions;
	}
	
	protected function _getCommentFetchOptions()
	{
		$commentFetchOptions = array(
			'join' => XenGallery_Model_Comment::FETCH_USER | XenGallery_Model_Comment::FETCH_CATEGORY | XenGallery_Model_Comment::FETCH_MEDIA
		);
		
		if (XenForo_Visitor::getInstance()->hasPermission('xengallery', 'viewDeletedComments'))
		{
			$commentFetchOptions['join'] |= XenGallery_Model_Comment::FETCH_DELETION_LOG;
		}

		return $commentFetchOptions;
	}		
	
	/**
	 * @return XenGallery_Model_Category
	 */
	protected function _getCategoryModel()
	{
		return $this->_controller->getModelFromCache('XenGallery_Model_Category');
	}
	
	/**
	 * @return XenGallery_Model_Media
	 */	
	protected function _getMediaModel()
	{
		return $this->_controller->getModelFromCache('XenGallery_Model_Media');
	}	
	
	/**
	 * @return XenForo_Model_Attachment
	 */	
	protected function _getAttachmentModel()
	{
		return $this->_controller->getModelFromCache('XenForo_Model_Attachment');
	}

	/**
	 * @return XenGallery_Model_Comment
	 */	
	protected function _getCommentModel()
	{
		return $this->_controller->getModelFromCache('XenGallery_Model_Comment');
	}
}