<?php

/**
 * Controller for XenGallery categories
 */
class XenGallery_ControllerAdmin_Category extends XenForo_ControllerAdmin_Abstract
{
	protected function _preDispatch($action)
	{
		$this->assertAdminPermission('manageXenGallery');
	}

	/**
	 * Lists all categories.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionIndex()
	{
		$viewParams = array(
			'categories' => $this->_getCategoryModel()->getCategoryStructure()
		);
		
		return $this->responseView('XenGallery_ViewAdmin_Category_List', 'xengallery_category_list', $viewParams);
	}

	/**
	 * Gets the category add/edit form response.
	 *
	 * @param array $category
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	protected function _getCategoryAddEditResponse(array $category)
	{
		$categories = $this->_getCategoryModel()->getCategoryStructure();
		
		$userGroups = $this->_getUserGroupModel()->getAllUserGroups();
		if (!empty($category['xengallery_category_id']))
		{
			$selUserGroupIds = unserialize($category['upload_user_groups']);
			
			if (is_bool($selUserGroupIds))
			{
				$selUserGroupIds = array();
			}
			
			if (in_array(-1, $selUserGroupIds))
			{
				$allUserGroups = true;
				$selUserGroupIds = array_keys($userGroups);
			}
			else
			{
				$allUserGroups = false;
			}
			
			$allowedTypes = unserialize($category['allowed_types']);
			
			if (is_bool($allowedTypes))
			{
				$allowedTypes = array();
			}
			
			if (in_array('all', $allowedTypes))
			{
				$allMediaTypes = true;
				$allowedTypes = array('image_upload', 'video_embed');
			}
			else
			{
				$allMediaTypes = false;
			}
		}
		else
		{
			$allUserGroups = true;
			$selUserGroupIds = array_keys($userGroups);
			
			$allMediaTypes = true;
			$allowedTypes = array('image_upload', 'video_embed');
		}		

		$viewParams = array(
			'category' => $category,
			'categories' => $categories,
			
			'userGroups' => $userGroups,
			'allUserGroups' => $allUserGroups,
			'selUserGroupIds' => $selUserGroupIds,
			
			'allMediaTypes' => $allMediaTypes,
			'allowedTypes' => $allowedTypes
		);
		return $this->responseView('XenGallery_ViewAdmin_Category_Edit', 'xengallery_category_edit', $viewParams);
	}

	/**
	 * Displays a form to create a new category.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionAdd()
	{
		$parentCategoryId = $this->_input->filterSingle('parent', XenForo_Input::UINT);
		
		return $this->_getCategoryAddEditResponse(array(
			'display_order' => 1,
			'parent_category_id' => $parentCategoryId
		));
	}

	/**
	 * Displays a form to edit an existing category.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionEdit()
	{
		$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
		$category = $this->_getCategoryOrError($categoryId);

		return $this->_getCategoryAddEditResponse($category);
	}

	/**
	 * Updates an existing media site or inserts a new one.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionSave()
	{
		$this->_assertPostOnly();

		$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
		
		$dwInput = $this->_input->filter(array(
			'category_title' => XenForo_Input::STRING,
			'category_description' => XenForo_Input::STRING,
			'parent_category_id' => XenForo_Input::UINT,			
			'display_order' => XenForo_Input::UINT,
			'usable_user_group_type' => XenForo_Input::STRING,
			'user_group_ids' => array(XenForo_Input::UINT, 'array' => true),
			'usable_media_type' => XenForo_Input::STRING,
			'media_type_checkboxes' => array(XenForo_Input::STRING, 'array' => true)
		));
		
		if ($dwInput['usable_user_group_type'] == 'all')
		{
			$allowedGroupIds = array(-1); // -1 is a sentinel for all groups
		}
		else
		{
			$allowedGroupIds = $dwInput['user_group_ids'];
		}
		
		if ($dwInput['usable_media_type'] == 'all')
		{
			$allowedMediaTypes = array('all');
		}
		else
		{
			$allowedMediaTypes = $dwInput['media_type_checkboxes'];
		}
		
		$dwInput['upload_user_groups'] = $allowedGroupIds;
		$dwInput['allowed_types'] = $allowedMediaTypes;
		
		unset($dwInput['user_group_ids']);
		unset($dwInput['usable_user_group_type']);
		unset($dwInput['media_type_checkboxes']);
		unset($dwInput['usable_media_type']);

		$dw = XenForo_DataWriter::create('XenGallery_DataWriter_Category');
		if ($categoryId)
		{
			$dw->setExistingData($categoryId);
		}
		$dw->bulkSet($dwInput);
		$dw->save();

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildAdminLink('xengallery-categories') . $this->getLastHash($dw->get('xengallery_category_id'))
		);
	}

	/**
	 * Deletes the specified category
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionDelete()
	{
		if ($this->isConfirmedPost())
		{
			return $this->_deleteData(
				'XenGallery_DataWriter_Category', 'xengallery_category_id',
				XenForo_Link::buildAdminLink('xengallery-categories')
			);
		}
		else // show confirmation dialog
		{
			$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
			
			$viewParams = array(
				'category' => $this->_getCategoryOrError($categoryId)
			);
			return $this->responseView('XenGallery_ViewAdmin_Category_Delete', 'xengallery_category_delete', $viewParams);
		}
	}

	/**
	 * Gets the specified record or errors.
	 *
	 * @param string $categoryId
	 *
	 * @return array
	 */
	protected function _getCategoryOrError($categoryId)
	{
		$category = $this->_getCategoryModel()->getCategoryById($categoryId);
		if (!$category)
		{
			throw $this->responseException($this->responseError(new XenForo_Phrase('requested_category_not_found'), 404));
		}

		return $category;
	}

	/**
	 * @return XenGallery_Model_Category
	 */
	protected function _getCategoryModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Category');
	}
	
	/**
	 * @return XenForo_Model_UserGroup
	 */
	protected function _getUserGroupModel()
	{
		return $this->getModelFromCache('XenForo_Model_UserGroup');
	}	
}