<?php

class Brivium_Credits_Model_Transaction extends XenForo_Model
{
	const FETCH_TRANSACTION_CREATE    			= 0x02;
	
	const FETCH_TRANSACTION_OWNER    			= 0x04;
	const FETCH_TRANSACTION_USERACTION    		= 0x05;
	const FETCH_TRANSACTION_FULL    			= 0x07;
	
	public function getAllTransactions(array $fetchOptions = array())
	{
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		$joinOptions = $this->prepareTransactionFetchOptions($fetchOptions);

		return $this->fetchAllKeyed($this->limitQueryResults('
				SELECT transaction.*
					' . $joinOptions['selectFields'] . '
				FROM xf_credits_transaction AS transaction
				' . $joinOptions['joinTables'] . '
				' . $whereClause . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'user_id');
	}
	/**
	*	get Category by its id
	* 	@param integer $transactionId
	* 	@param array $fetchOptions Collection of options related to fetching
	*	@return array|false Category info
	*/
	public function getTransactionById($transactionId,$fetchOptions = array()){
		$joinOptions = $this->prepareTransactionFetchOptions($fetchOptions);
		return $this->_getDb()->fetchRow('
			SELECT transaction.*
			' .$joinOptions['selectFields']. '
			FROM xf_credits_transaction AS transaction
			' .$joinOptions['joinTables']. '
			WHERE transaction.transaction_id = ?
		',$transactionId);
	}
	/**
	*	Gets multi transactions.
	*
	*	@param array $transactionIds
	*	@param array $fetchOptions Collection of options related to fetching
	*	
	*	@return array Format: [transaction id] => info
	*/
	public function getTransactionsByIds(array $transactionIds)
	{
		if (!$transactionIds)
		{
			return array();
		}
		return $this->fetchAllKeyed('
			SELECT transaction.*
			FROM xf_credits_transaction AS transaction
			WHERE transaction.transaction_id IN (' . $this->_getDb()->quote($transactionIds) . ')
		', 'transaction_id');
	}
	
	/**
	 * Prepares a collection of transaction fetching related conditions into an SQL clause
	 *
	 * @param array $conditions List of conditions
	 * @param array $fetchOptions Modifiable set of fetch options (may have joins pushed on to it)
	 *
	 * @return string SQL clause (at least 1=1)
	 */
	public function prepareTransactionConditions(array $conditions, array &$fetchOptions)
	{
		$sqlConditions = array();
		$db = $this->_getDb();

		if (!empty($conditions['transaction_id']))
		{
			if (is_array($conditions['transaction_id']))
			{
				$sqlConditions[] = 'transaction.transaction_id IN (' . $db->quote($conditions['transaction_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'transaction.transaction_id = ' . $db->quote($conditions['transaction_id']);
			}
		}
		if (!empty($conditions['action_id']))
		{
			if (is_array($conditions['action_id']))
			{
				$sqlConditions[] = 'transaction.action_id IN (' . $db->quote($conditions['action_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'transaction.action_id = ' . $db->quote($conditions['action_id']);
			}
		}
		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'transaction.user_id IN (' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'transaction.user_id = ' . $db->quote($conditions['user_id']);
			}
		}
		if (!empty($conditions['user_action_id']))
		{
			$sqlConditions[] = 'transaction.user_action_id = ' . $db->quote($conditions['user_action_id']);
		}
		if (!empty($conditions['amount']) && is_array($conditions['amount']))
		{
			list($operator, $cutOff) = $conditions['amount'];

			$this->assertValidCutOffOperator($operator);
			$sqlConditions[] = "transaction.amount $operator " . $db->quote($cutOff);
		}
		if (!empty($conditions['transaction_date']) && is_array($conditions['transaction_date']))
		{
			list($operator, $cutOff) = $conditions['transaction_date'];

			$this->assertValidCutOffOperator($operator);
			$sqlConditions[] = "transaction.transaction_date $operator " . $db->quote($cutOff);
		}
		if (!empty($conditions['start']))
		{
			$sqlConditions[] = 'transaction.transaction_date >= ' . $db->quote($conditions['start']);
		}

		if (!empty($conditions['end']))
		{
			$sqlConditions[] = 'transaction.transaction_date <= ' . $db->quote($conditions['end']);
		}

		return $this->getConditionsForClause($sqlConditions);
	}
	public function prepareTransactionFetchOptions(array $fetchOptions)
	{
		$selectFields = '';
		$joinTables = '';
		$orderBy = '';
		$orderBySecondary = '';
		if (!empty($fetchOptions['order']))
		{
			switch ($fetchOptions['order'])
			{
				case 'action_id':
				case 'user_id':
				case 'owner_id':
				case 'amount':
					$orderBy = 'transaction.' . $fetchOptions['order'];
					$orderBySecondary = ', transaction.transaction_date DESC';
					break;
				default:
					$orderBy = 'transaction.transaction_date';	
			}
			if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'desc')
			{
				$orderBy .= ' DESC';
			}
			else
			{
				$orderBy .= ' ASC';
			}
			$orderBy .= $orderBySecondary;
		}

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_TRANSACTION_CREATE)
			{
				$selectFields .= ', user_create.username AS username';
				$joinTables .= '
					LEFT JOIN xf_user AS user_create ON
						(user_create.user_id = transaction.user_id)';
			}
			if($fetchOptions['join'] & self::FETCH_TRANSACTION_OWNER)
			{
				$selectFields .= ', owner.username AS owner_name';
				$joinTables .= '
					LEFT JOIN xf_user AS owner ON
						(owner.user_id = transaction.owner_id)';
			}
			if($fetchOptions['join'] & self::FETCH_TRANSACTION_USERACTION)
			{
				$selectFields .= ', user_action.username AS user_action_name';
				$joinTables .= '
					LEFT JOIN xf_user AS user_action ON
						(user_action.user_id = transaction.user_action_id)';
			}
			
		}
		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause'  => ($orderBy ? "ORDER BY $orderBy" : '')
		);
	}
	
	/**
	 * Gets transactions that match the given conditions.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 * @param array $fetchOptions Collection of options that relate to fetching
	 *
	 * @return array Format: [transaction id] => info
	 */
	public function getTransactions(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);

		$sqlClauses = $this->prepareTransactionFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		return $this->fetchAllKeyed($this->limitQueryResults(			'
				SELECT transaction.*
					' . $sqlClauses['selectFields'] . '
				FROM xf_credits_transaction AS transaction
				' . $sqlClauses['joinTables'] . '
				WHERE ' . $whereConditions . '
				' . $sqlClauses['orderClause'] . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'transaction_id');
	}
	public function getTopSpentTransactions(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		return $this->fetchAllKeyed($this->limitQueryResults(			'
				SELECT user_id, SUM(amount) AS credits
				FROM xf_credits_transaction AS transaction
				WHERE ' . $whereConditions . ' AND transaction.amount < 0 
				GROUP BY user_id
				ORDER BY credits ASC
			', $limitOptions['limit'], $limitOptions['offset']
		), 'user_id');
	}
	public function getTopEarnedTransactions(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		return $this->fetchAllKeyed($this->limitQueryResults(			'
				SELECT user_id, SUM(amount) AS credits
				FROM xf_credits_transaction AS transaction
				WHERE ' . $whereConditions . ' AND transaction.amount > 0 
				GROUP BY user_id
				ORDER BY credits DESC
			', $limitOptions['limit'], $limitOptions['offset']
		), 'user_id');
	}
	
	public function getTotalSpentCredits(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		return $this->_getDb()->fetchOne($this->limitQueryResults(			'
				SELECT SUM(amount) AS credits
				FROM xf_credits_transaction AS transaction
				WHERE ' . $whereConditions . ' AND transaction.amount < 0
			', $limitOptions['limit'], $limitOptions['offset']
		));
	}
	public function getTotalEarnedCredits(array $conditions, array $fetchOptions = array())
	{
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		return $this->_getDb()->fetchOne($this->limitQueryResults(			'
				SELECT SUM(amount) AS credits
				FROM xf_credits_transaction AS transaction
				WHERE ' . $whereConditions . ' AND transaction.amount > 0
			', $limitOptions['limit'], $limitOptions['offset']
		));
	}
	
	/**
	 * Gets the count of transactions with the specified criteria.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 *
	 * @return integer
	 */
	public function countTransactions(array $conditions)
	{
		$fetchOptions = array();
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xf_credits_transaction AS transaction
			WHERE ' . $whereConditions . '
		');
	}
	
	/**
	 * Gets the count of day start transactions.
	 *
	 * @param array $conditions Conditions to apply to the fetching
	 *
	 * @return integer
	 */
	public function firstTimeTransactions(array $conditions)
	{
		$fetchOptions = array();
		$whereConditions = $this->prepareTransactionConditions($conditions, $fetchOptions);
		return $this->_getDb()->fetchOne('
			SELECT transaction_date
			FROM xf_credits_transaction AS transaction
			WHERE ' . $whereConditions . '
			ORDER BY transaction.transaction_date ASC
			LIMIT 1
		');
	}
	
	/**
	 * Delete all transaction data.
	 */
	public function deleteAllTransaction(){
		$db = $this->_getDb();
		$db->query('TRUNCATE TABLE xf_credits_transaction');
		XenForo_Db::commit($db);
	}
	public function prepareTransactions(array $transactions)
	{
		if(!$transactions) return array();
		foreach($transactions AS &$transaction){
			$transaction = $this->prepareTransaction($transaction);
		}
		return $transactions;
	}
	public function prepareTransaction(array $tran)
	{
		if(!$tran) return array();
		
		$negate = ( $tran['negate'] ? array('<i>', '</i>', '<s>', '</s>') : array('', '', '', '') );
		$which = 'earned';
		if ($tran['amount'] < 0)
		{
			$which = 'spent';
			//$tran['amount'] *= -1;
		}
		$tran['action'] = $negate[2] . new XenForo_Phrase('BRC_action_' . $tran['action_id']) . $negate[3];
		if ($tran['extra_data'])
		{
			$tran['extraData'] = @unserialize($tran['extraData']);
			if (!is_array($tran['extraData']))
			{
				$tran['extraData'] = array();
			}
			if(!empty($tran['extraData']['reverted']))$tran['action'] = $negate[2] . new XenForo_Phrase('BRC_action_' . $tran['action_id'].'_reverted') . $negate[3];
		}
		
		
		if(!$tran['user_action_id'])$tran['user_action_name'] = 'System';
		
		//$amount = XenForo_Locale::numberFormat($tran['amount'],2);
		$currency = 'point';
		$tran['amount_phrase'] = $negate[0] .  new XenForo_Phrase('BRC_transaction_' . $which, array('amount'=>Brivium_Credits_Listener_Helpers::helperCurrencyFormat($tran['amount']))). $negate[1];
		
		return $tran;
	}
	
	/**
	 * Clear out of date transaction.
	 */
	public function transactionClear()
	{
		$db = $this->_getDb();
		XenForo_Db::beginTransaction($db);
		$transactionsHistory = XenForo_Application::$time - (XenForo_Application::get('options')->BRC_transactionsHistory * 86400);
		$db->delete('xf_credits_transaction', 'transaction_date < ' . $transactionsHistory);
		XenForo_Db::commit($db);
	}
	
}

?>