<?php

abstract class Waindigo_Install
{
	/**
	 * @var Zend_Db_Adapter_Abstract
	 */
	protected $_db;

	protected $_prerequisites;
	protected $_fieldNameChanges;
	protected $_tables;
	protected $_tableChanges;
	protected $_contentTypes;
	protected $_contentTypeFields;
	protected $_nodeTypes;
	protected $_userFields;
	protected $_primaryKeys;
	protected $_uniqueKeys;
	protected $_keys;
	protected $_fields;

	protected static $_tablesList;

	protected $_minVersionId = 1000000;
	protected $_minVersionString = '1.0.0';
	protected $_resourceManagerUrl = '';
	protected $_addOnData = array();

	/**
	 * Standard approach to caching other model objects for the lifetime of the model.
	 *
	 * @var array
	 */
	protected $_modelCache = array();

	/**
	 * Gets the specified model object from the cache. If it does not exist,
	 * it will be instantiated.
	 *
	 * @param string $class Name of the class to load
	 *
	 * @return XenForo_Model
	 */
	public function getModelFromCache($class)
	{
		if (!isset($this->_modelCache[$class])) {
			$this->_modelCache[$class] = XenForo_Model::create($class);
		}

		return $this->_modelCache[$class];
	} /* END Waindigo_Install::getModelFromCache */

	/**
	 * Gets the autoloader's root directory.
	 *
	 * @return string
	 */
	public function getRootDir()
	{
		return XenForo_Autoloader::getInstance()->getRootDir();
	} /* END Waindigo_Install::getRootDir */

	/**
	 * At some point this will be made final, but old add-ons still extend it at present.
	 */
	public static function install()
	{
		$addOnData = func_get_arg(1);
		if (class_exists($addOnData['addon_id']."_Install")) {
			$installer = self::create($addOnData['addon_id']."_Install");
			$installer->_install($addOnData);
		}
	} /* END Waindigo_Install::install */

	/**
	 * At some point this will be made final, but old add-ons still extend it at present.
	 */
	public static function uninstall()
	{
		$addOnData = func_get_arg(0);
		if (class_exists($addOnData['addon_id']."_Install")) {
			$uninstaller = self::create($addOnData['addon_id']."_Install");
			$uninstaller->_uninstall($addOnData);
		}
	} /* END Waindigo_Install::uninstall */

	/**
	 * Factory method to get the named installer. The class must exist or be autoloadable
	 * or an exception will be thrown.
	 *
	 * @param string Class to load
	 *
	 * @return Waindigo_Install
	 */
	public static function create($class)
	{
		$createClass = XenForo_Application::resolveDynamicClass($class, 'installer_waindigo');
		if (!$createClass) {
			throw new XenForo_Exception("Invalid installer '$class' specified");
		}

		return new $createClass;
	} /* END Waindigo_Install::create */

	protected final function _install(array $addOnData = array())
	{
	    $this->_addOnData = $addOnData;
	    $this->_checkXenForoVersion();
	    if (!empty($this->_prerequisites)) {
    	    $this->_checkPrerequisites($this->_prerequisites);
	    }
		$this->_preInstallBeforeTransaction();
		$this->_db->beginTransaction();
		$this->_preInstall();
		if (!empty($this->_fieldNameChanges)) {
			$this->_makeFieldNameChanges($this->_fieldNameChanges);
		}
		if (!empty($this->_tables)) {
			$this->_createTables($this->_tables);
		}
		if (!empty($this->_tableChanges)) {
			$this->_makeTableChanges($this->_tableChanges);
		}
		if (!empty($this->_contentTypeFields)) {
			$this->_insertContentTypeFields($this->_contentTypeFields);
		}
		if (!empty($this->_contentTypes) || !empty($this->_contentTypeFields)) {
			$this->_insertContentTypes($this->_contentTypes);
		}
		if (!empty($this->_nodeTypes)) {
			$this->_insertNodeTypes($this->_nodeTypes);
		}
		if (!empty($this->_userFields)) {
			$this->_createUserFields($this->_userFields);
		}
		if (!empty($this->_primaryKeys)) {
			$this->_addPrimaryKeys($this->_primaryKeys);
		}
		if (!empty($this->_uniqueKeys)) {
			$this->_addUniqueKeys($this->_uniqueKeys);
		}
		if (!empty($this->_keys)) {
			$this->_addKeys($this->_keys);
		}
		if (!empty($this->_fields)) {
			$this->_insertFields($this->_fields);
		}
		if ($this->_resourceManagerUrl) {
		    $this->_updateAddOnInstaller($this->_resourceManagerUrl);
		}
		$this->_postInstall();
		$this->_db->commit();
		$this->_postInstallAfterTransaction();
	} /* END Waindigo_Install::_install */

	protected final function _uninstall(array $addOnData = array())
	{
	    $this->_addOnData = $addOnData;
		$this->_preUninstallBeforeTransaction();
		$this->_db->beginTransaction();
		$this->_preUninstall();
		if (!empty($this->_tables)) {
			$this->_dropTables($this->_tables);
		}
		if (!empty($this->_tableChanges)) {
			$this->_dropTableChanges($this->_tableChanges);
		}
		if (!empty($this->_contentTypeFields)) {
			$this->_deleteContentTypes($this->_contentTypeFields);
		}
		if (!empty($this->_contentTypes) || !empty($this->_contentTypeFields)) {
			$this->_deleteContentTypes($this->_contentTypes);
		}
		if (!empty($this->_nodeTypes)) {
			$this->_deleteNodeTypes($this->_nodeTypes);
		}
		if (!empty($this->_userFields)) {
			$this->_dropUserFields($this->_userFields);
		}
		if (!empty($this->_fields)) {
			$this->_deleteFields($this->_fields);
		}
		$this->_postUninstall();
		$this->_db->commit();
		$this->_postUninstallAfterTransaction();
	} /* END Waindigo_Install::_uninstall */

	public function __construct()
	{
		$this->_db = XenForo_Application::get('db');

		$this->_prerequisites = $this->_getPrerequisites();

		$this->_fieldNameChanges = $this->_getFieldNameChanges();
		$this->_tables = $this->_getTables();
		$this->_tableChanges = $this->_getAllTableChanges();
		$this->_contentTypes = $this->_getContentTypes();
		$this->_contentTypeFields = $this->_getContentTypeFields();
		$this->_nodeTypes = $this->_getNodeTypes();
		$this->_userFields = $this->_getUserFields();
		$this->_primaryKeys = $this->_getPrimaryKeys();
		$this->_uniqueKeys = $this->_getUniqueKeys();
		$this->_keys = $this->_getKeys();
		$this->_fields = $this->_getFields();
	} /* END Waindigo_Install::__construct */

	/**
	 * @return array [addon id] => version id
	 */
	protected function _getPrerequisites()
	{
	    return array();
	} /* END Waindigo_Install::_getPrerequisites */

	/**
	 * @param string|array|null $addOnIds
	 */
	public static function getPrerequisites($addOnIds = null)
	{
	    $prerequisites = $this->_prerequisites;
	    if (!$addOnIds) {
	        return $prerequisites;
	    } else if (is_array($addOnIds)) {
	        return XenForo_Application::arrayFilterKeys($prerequisites, $addOnIds);
	    } else {
	        return (isset($prerequisites[$addOnIds]) ? $prerequisites[$addOnIds] : array());
	    }
	} /* END Waindigo_Install::getPrerequisites */

	/**
	 * @return array
	 */
	protected function _getFieldNameChanges()
	{
		return array();
	} /* END Waindigo_Install::_getFieldNameChanges */

	/**
	 * Gets the tables (with fields) to be created for this add-on.
	 *
	 * @return array Format: [table name] => fields
	 */
	protected function _getTables()
	{
		return array();
	} /* END Waindigo_Install::_getTables */

	/**
	 * Gets the field changes (grouped by table) to be made for this add-on.
	 *
	 * @return array Format: [table name] => field changes
	 */
	protected function _getTableChanges()
	{
		return array();
	} /* END Waindigo_Install::_getTableChanges */

	/**
	 * Gets the field changes to be made only if the specified add-ons are installed.
	 *
	 * @return array Format: [add-on id] => table changes
	 */
	protected function _getAddOnTableChanges()
	{
	    return array();
	} /* END Waindigo_Install::_getAddOnTableChanges */

	/**
	 * @return array
	 */
	protected function _getContentTypes()
	{
		return array();
	} /* END Waindigo_Install::_getContentTypes */

	/**
	 * @return array
	 */
	protected function _getContentTypeFields()
	{
		return array();
	} /* END Waindigo_Install::_getContentTypeFields */

	/**
	 * @return array
	 */
	protected function _getNodeTypes()
	{
		return array();
	} /* END Waindigo_Install::_getNodeTypes */

	/**
	 * @return array
	 */
	protected function _getUserFields()
	{
		return array();
	} /* END Waindigo_Install::_getUserFields */

	/**
	 * @return array
	 */
	protected function _getPrimaryKeys()
	{
		return array();
	} /* END Waindigo_Install::_getPrimaryKeys */

	/**
	 * @return array
	 */
	protected function _getUniqueKeys()
	{
		return array();
	} /* END Waindigo_Install::_getUniqueKeys */

	/**
	 * @return array
	 */
	protected function _getKeys()
	{
		return array();
	} /* END Waindigo_Install::_getKeys */

	/**
	 * @return array
	 */
	protected function _getFields()
	{
		return array();
	} /* END Waindigo_Install::_getFields */

	protected function _checkXenForoVersion()
	{
	    if (XenForo_Application::$versionId < $this->_minVersionId) {
			throw new XenForo_Exception('Minimum XenForo version of ' . $this->_minVersionString . ' required.');
		}
	} /* END Waindigo_Install::_checkXenForoVersion */

	/**
	 * @param array $prerequisites
	 */
	protected function _checkPrerequisites(array $prerequisites)
	{
        /* @var $addOnModel XenForo_Model_AddOn */
        $addOnModel = $this->getModelFromCache('XenForo_Model_AddOn');
	    foreach ($prerequisites as $addOnId => $versionId) {
	        $addOn = $addOnModel->getAddOnById($addOnId);
	        if (!$addOn){
	            throw new XenForo_Exception('Required add-ons not installed.');
	        }
	        if ($addOn['version_id'] < $versionId) {
	            throw new XenForo_Exception('Required add-ons not up to date.');
	        }
	    }
	} /* END Waindigo_Install::_checkPrerequisites */

	/**
	 * @param array $fieldNameChanges
	 */
	protected function _makeFieldNameChanges(array $fieldNameChanges)
	{
		foreach ($fieldNameChanges as $tableName => $rows) {
			if ($this->_isTableExists($tableName)) {
				$describeTable = $this->_db->describeTable($tableName);
				$keys = array_keys($describeTable);
				$sql = "ALTER TABLE `".$tableName."` ";
				$sqlAdd = array();
				foreach ($rows as $oldFieldName => $newField) {
					if (in_array($oldFieldName, $keys)) {
						$sqlAdd[] = "CHANGE `".$oldFieldName."` ".$newField;
					}
				}
				$sql .= implode(", ", $sqlAdd);
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_makeFieldNameChanges */

	/**
	 * @param array $tables
	 */
	protected function _createTables(array $tables)
	{
		foreach ($tables as $tableName => $rows) {
			if (!$this->_isTableExists($tableName)) {
				$sql = "CREATE TABLE IF NOT EXISTS `".$tableName."` (";
				$sqlRows = array();
				foreach ($rows as $rowName => $rowParams) {
					$sqlRows[] = "`".$rowName."` ".$rowParams;
				}
				$sql.= implode(",", $sqlRows);
				$sql.= ") ENGINE = InnoDB CHARACTER SET utf8 COLLATE utf8_general_ci";
				$this->_db->query($sql);
				if (self::$_tablesList) {
				    self::$_tablesList[] = strtolower($tableName);
				}
			} else {
				$sql = "ALTER TABLE `".$tableName."` ENGINE = InnoDB CHARACTER SET utf8 COLLATE utf8_general_ci";
			}
		}

		$this->_makeTableChanges($tables);
	} /* END Waindigo_Install::_createTables */

	/**
	 * @param array $tableChanges
	 */
	protected function _makeTableChanges(array $tableChanges)
	{
		foreach ($tableChanges as $tableName => $rows) {
			if ($this->_isTableExists($tableName)) {
				$describeTable = $this->_db->describeTable($tableName);
				$keys = array_keys($describeTable);
				$sql = "ALTER IGNORE TABLE `".$tableName."` ";
				$sqlAdd = array();
				foreach ($rows as $rowName => $rowParams) {
					if (strpos($rowParams, 'PRIMARY KEY') !== false) {
						if ($this->_getExistingPrimaryKeys($tableName)) {
							$sqlAdd[] = "DROP PRIMARY KEY ";
						}
					}
					if (in_array($rowName, $keys)) {
						$sqlAdd[] = "CHANGE `".$rowName."` `".$rowName."` ".$rowParams;
					} else {
						$sqlAdd[] = "ADD `".$rowName."` ".$rowParams;
					}
				}
				$sql .= implode(", ", $sqlAdd);
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_makeTableChanges */

	/**
	 * @param string $tableName
	 * @return array $existingPrimaryKeys
	 */
	protected function _getExistingPrimaryKeys($tableName)
	{
		$primaryKeys = array();
	    if ($this->_isTableExists($tableName)) {
    	    $columns = $this->_db->describeTable($tableName);
    		foreach ($columns as $columnName => $column) {
    			if ($column['PRIMARY']) {
    				$primaryKeys[] = $columnName;
    			}
    		}
		}
		return $primaryKeys;
	} /* END Waindigo_Install::_getExistingPrimaryKeys */

	/**
	 * @param array $primaryKeys
	 */
	protected function _addPrimaryKeys(array $primaryKeys)
	{
		foreach ($primaryKeys as $tableName => $primaryKey) {
			$oldKey = $this->_getExistingPrimaryKeys($tableName);
			$keyDiff = array_diff($primaryKey, $oldKey);
			if (!empty($keyDiff)) {
				$sql = "ALTER TABLE `".$tableName."`
					". (empty($oldKey) ? "": "DROP PRIMARY KEY, ") ."
					ADD PRIMARY KEY(".implode(",", $primaryKey).")";
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_addPrimaryKeys */

    /**
     * @param string $tableName
     * @return array $existingKeys
     */
	protected function _getExistingKeys($tableName)
	{
		$keys = array();
	    if ($this->_isTableExists($tableName)) {
    	    $columns = $this->_db->describeTable($tableName);
    		$indexes = $this->_db->fetchAll('SHOW INDEXES FROM  `'.$tableName.'`');
    		foreach ($indexes as $index) {
    			$keys[$index['Key_name']] = $index;
    		}
	    }
		return $keys;
	} /* END Waindigo_Install::_getExistingKeys */

	/**
	 * @param array $uniqueKeys
	 */
	protected function _addUniqueKeys(array $uniqueKeys)
	{
		foreach ($uniqueKeys as $tableName => $uniqueKey) {
			$oldKeys = $this->_getExistingKeys($tableName);
			foreach ($uniqueKey as $keyName => $keyColumns) {
				$sql = "ALTER TABLE `".$tableName."`
					". (!isset($oldKeys[$keyName]) ? "": "DROP INDEX `" . $keyName . "`, ") ."
					ADD UNIQUE `" . $keyName . "` (" . implode(",", $keyColumns) . ")";
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_addUniqueKeys */

	/**
	 * @param array $keys
	 */
	protected function _addKeys(array $keys)
	{
		foreach ($keys as $tableName => $key) {
    	    if ($this->_isTableExists($tableName)) {
    		    $oldKeys = $this->_getExistingKeys($tableName);
    			foreach ($key as $keyName => $keyColumns) {
    				$sql = "ALTER TABLE `".$tableName."`
    					". (!isset($oldKeys[$keyName]) ? "": "DROP INDEX `" . $keyName . "`, ") ."
    					ADD INDEX `" . $keyName . "` (" . implode(",", $keyColumns) . ")";
    				$this->_db->query($sql);
    			}
    		}
		}
	} /* END Waindigo_Install::_addKeys */

    /**
     * @param array $tables
     */
	protected function _dropTables(array $tables)
	{
		foreach ($tables as $tableName => $rows) {
			$sql = "DROP TABLE IF EXISTS `".$tableName."` ";
			$this->_db->query($sql);
			if (self::$_tablesList && in_array($tableName, self::$_tablesList)) {
			    unset(self::$_tablesList[array_search($tableName, self::$_tablesList)]);
			}
		}
	} /* END Waindigo_Install::_dropTables */

	/**
	 * @param array $tableChanges
	 */
	protected function _dropTableChanges(array $tableChanges)
	{
		foreach ($tableChanges as $tableName => $rows) {
		    if ($this->_isTableExists($tableName)) {
    			$keys = array_keys($this->_db->describeTable($tableName));
    			foreach ($rows as $rowName => $rowParams) {
    				if (in_array($rowName, $keys)) {
    					$sql = "ALTER TABLE `".$tableName."` DROP `".$rowName;
    					$this->_db->query($sql);
    				}
    			}
		    }
		}
	} /* END Waindigo_Install::_dropTableChanges */

	/**
	 * @param array $inserts
	 */
	protected function _insertFields(array $inserts)
	{
		foreach ($inserts as $insert) {
			if (isset($insert['table_name'], $insert['data_writer'])) {
				$dw = XenForo_DataWriter::create($insert['data_writer']);
				if (isset($insert['primary_fields'])) {
					$strSql = "SELECT count(*) FROM `".$insert['table_name']."` ";
					$whereClauses = array();
					foreach ($insert['primary_fields'] as $fieldName => $fieldValue) {
						$whereClauses[] = "`".$fieldName."` = '".$fieldValue."'";
					}
					if (!empty($whereClauses)) {
						$strSql.= "WHERE ".implode(" AND ", $whereClauses)." ";
					}
					if ($this->_db->fetchOne($strSql)) {
						$dw->setExistingData($insert['primary_fields']);
					}
				}
				$dw->bulkSet($insert['primary_fields']);
				if (isset($insert['fields'])) {
					$dw->bulkSet($insert['fields']);
				}
				$dw->save();
			}
		}
	} /* END Waindigo_Install::_insertFields */

	/**
	 * @param array $inserts
	 */
	protected function _deleteFields(array $inserts)
	{
		foreach ($inserts as $insert) {
			if (isset($insert['data_writer'])) {
				$dw = XenForo_DataWriter::create($insert['data_writer']);
				if (isset($insert['primary_fields'])) {
					$dw->setExistingData($insert['primary_fields']);
				}
				$dw->delete();
			}
		}
	} /* END Waindigo_Install::_deleteFields */

	/**
	 * @param array $userFields
	 */
	protected function _createUserFields(array $userFields)
	{
		foreach ($userFields as $fieldId => $fields)
		{
			$dw = XenForo_DataWriter::create('XenForo_DataWriter_UserField');
			if (!$dw->setExistingData($fieldId)) {
				$dw->set('field_id', $fieldId);
			}
			$dw->bulkSet($fields);
			$dw->save();
		}
	} /* END Waindigo_Install::_createUserFields */

	/**
	 * @param array $userFields
	 */
	protected function _dropUserFields(array $userFields)
	{
		foreach ($userFields as $fieldId => $fields) {
			$dw = XenForo_DataWriter::create('XenForo_DataWriter_UserField');
			$dw->setExistingData($fieldId);
			$dw->delete();
		}
	} /* END Waindigo_Install::_dropUserFields */

	/**
	 * @param array $contentTypes
	 */
	protected function _insertContentTypes(array $contentTypes)
	{
		foreach ($contentTypes as $contentType => $contentTypeParams) {
			if (isset($contentTypeParams['addon_id'])) {
				$addOnId = $contentTypeParams['addon_id'];
				$sql = "INSERT INTO xf_content_type (
							content_type,
							addon_id,
							fields
						) VALUES (
							'".$contentType."',
							'".$addOnId."',
							''
						) ON DUPLICATE KEY UPDATE
							addon_id = '".$addOnId."'";
				$this->_db->query($sql);

				if (isset($contentTypeParams['fields'])) {
    				$this->_insertContentTypeFields(array($contentType => $contentTypeParams['fields']));
				}
			}
		}
		XenForo_Model::create('XenForo_Model_ContentType')->rebuildContentTypeCache();
	} /* END Waindigo_Install::_insertContentTypes */

	/**
	 * @param array $contentTypes
	 */
	protected function _insertContentTypeFields(array $contentTypes)
	{
		foreach ($contentTypes as $contentType => $contentTypeFields) {
			foreach ($contentTypeFields as $fieldName => $fieldValue) {
				$sql = "INSERT INTO xf_content_type_field (
						content_type,
						field_name,
						field_value
					) VALUES (
						'".$contentType."',
						'".$fieldName."',
						'".$fieldValue."'
					) ON DUPLICATE KEY UPDATE
						field_value = '".$fieldValue."'";
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_insertContentTypeFields */

	/**
	 * @param array $contentTypes
	 */
	protected function _deleteContentTypes(array $contentTypes)
	{
		foreach ($contentTypes as $contentType => $contentTypeParams) {
			if (isset($contentTypeParams['addon_id'])) {
				$addOnId = $contentTypeParams['addon_id'];
				$sql = "DELETE FROM xf_content_type WHERE content_type = '".$contentType."' AND addon_id = '".$addOnId."'";
				$this->_db->query($sql);
				$sql = "DELETE FROM xf_content_type_field WHERE content_type = '".$contentType."'";
				$this->_db->query($sql);
			}
		}
		XenForo_Model::create('XenForo_Model_ContentType')->rebuildContentTypeCache();
	} /* END Waindigo_Install::_deleteContentTypes */

	/**
	 * @param array $contentTypes
	 */
	protected function _deleteContentTypeFields(array $contentTypes)
	{
		foreach ($contentTypes as $contentType => $contentTypeFields) {
			foreach ($contentTypeFields as $fieldName => $fieldValue) {
				$sql = "DELETE FROM xf_content_type_field WHERE content_type = '".$contentType."'
						AND field_name = '".$fieldName."' AND field_value = '".$fieldValue."'";
				$this->_db->query($sql);
			}
		}
	} /* END Waindigo_Install::_deleteContentTypeFields */

	/**
	 * @param array $nodeTypes
	 */
	protected function _insertNodeTypes(array $nodeTypes)
	{
		foreach ($nodeTypes as $nodeTypeId => $nodeTypeParams) {
			$sql = "INSERT INTO `xf_node_type` (
					`node_type_id`,
					`handler_class`,
					`controller_admin_class`,
					`datawriter_class`,
					`permission_group_id`,
					`public_route_prefix`
				) VALUES (
					'".$nodeTypeId."',
					'".$nodeTypeParams['handler_class']."',
					'".$nodeTypeParams['controller_admin_class']."',
					'".$nodeTypeParams['datawriter_class']."',
				 	'".$nodeTypeParams['permission_group_id']."',
				 	'".$nodeTypeParams['public_route_prefix']."'
				 ) ON DUPLICATE KEY UPDATE
					handler_class = '".$nodeTypeParams['handler_class']."',
					controller_admin_class = '".$nodeTypeParams['controller_admin_class']."',
					datawriter_class = '".$nodeTypeParams['datawriter_class']."',
					permission_group_id = '".$nodeTypeParams['permission_group_id']."',
					public_route_prefix = '".$nodeTypeParams['public_route_prefix']."'";
			$this->_db->query($sql);
		}
		XenForo_Model::create('XenForo_Model_Node')->rebuildNodeTypeCache();
	} /* END Waindigo_Install::_insertNodeTypes */

	/**
	 * @param array $nodeTypes
	 */
	protected function _deleteNodeTypes(array $nodeTypes)
	{
		foreach ($nodeTypes as $nodeTypeId => $nodeTypeParams) {
			$sql = "DELETE FROM `xf_node_type` WHERE `node_type_id` = '".$nodeTypeId."' ";
			$this->_db->query($sql);
			$sql = "DELETE FROM `xf_node` WHERE `node_type_id` = '".$nodeTypeId."' ";
			$this->_db->query($sql);
		}
		XenForo_Model::create('XenForo_Model_Node')->rebuildNodeTypeCache();
	} /* END Waindigo_Install::_deleteNodeTypes */

	protected function _updateAddOnInstaller($resourceUrl)
	{
	    if ($this->_addOnData) {
    	    $data = array(
                'addon_id' => $this->_addOnData['addon_id'],
                'update_url' => $resourceUrl,
                'check_updates' => 1,
                'last_checked' => XenForo_Application::$time,
                'latest_version' => $this->_addOnData['version_string']
    	    );

    	    try {
        	    $writer = XenForo_DataWriter::create('AddOnInstaller_DataWriter_Updater');
    	    } catch (Exception $e) {
    	        // do nothing
    	    }
    	    /* @var $addOnModel XenForo_Model_AddOn */
    	    $addOnModel = $this->getModelFromCache('XenForo_Model_AddOn');
    	    if (method_exists($addOnModel, 'isDwUpdate')) {
    	        if ($addOnModel->isDwUpdate($data['addon_id'])) {
    	            $writer->setExistingData($data['addon_id']);
    	        }
    	        $writer->bulkSet($data);
    	        $writer->save();
    	    }
	    }
	}

	protected function _preInstall()
	{
	} /* END Waindigo_Install::_preInstall */

	protected function _preInstallBeforeTransaction()
	{
	} /* END Waindigo_Install::_preInstallBeforeTransaction */

	protected function _preUninstall()
	{
	} /* END Waindigo_Install::_preUninstall */

	protected function _preUninstallBeforeTransaction()
	{
	} /* END Waindigo_Install::_preUninstallBeforeTransaction */

	protected function _postInstall()
	{
	} /* END Waindigo_Install::_postInstall */

	protected function _postInstallAfterTransaction()
	{
	} /* END Waindigo_Install::_postInstallAfterTransaction */

	protected function _postUninstall()
	{
	} /* END Waindigo_Install::_postUninstall */

	protected function _postUninstallAfterTransaction()
	{
	} /* END Waindigo_Install::_postUninstallAfterTransaction */

	/**
	 * @param string $tableName
	 * @return boolean
	 */
	protected function _isTableExists($tableName)
	{
	    if (!self::$_tablesList) {
	        self::$_tablesList = array_map('strtolower', $this->_db->listTables());
	    }
	    return in_array(strtolower($tableName), self::$_tablesList);
	} /* END Waindigo_Install::_isTableExists */

	/**
	 * @param string $addOnId
	 * @return array $tableChanges
	 */
	protected function getTableChanges($addOnId)
	{
	    $addOnTableChanges = $this->_getAddOnTableChanges();
	    if (isset($addOnTableChanges[$addOnId])) {
	        return $addOnTableChanges[$addOnId];
	    }
	    return array();
	} /* END Waindigo_Install::getTableChanges */

	/**
	 * @return array $tableChanges
	 */
	protected function _getAllTableChanges()
	{
	    /* @var $addOnModel XenForo_Model_AddOn */
	    $addOnModel = XenForo_Model::create('XenForo_Model_AddOn');
	    $addOns = $addOnModel->getAllAddOns();

	    $tableChanges = $this->_getTableChanges();
	    foreach ($addOns as $addOnId => $addOn) {
	        $tableChanges = array_merge(
	            $tableChanges, $this->getTableChanges($addOnId)
	        );
	    }

	    return $tableChanges;
	} /* END Waindigo_Install::getAllTableChanges */
}