<?php

class XenGallery_Model_Comment extends XenForo_Model
{
	const FETCH_USER        = 0x01;
	const FETCH_MEDIA	    = 0x02;
	const FETCH_ATTACHMENT  = 0x04;
	const FETCH_DELETION_LOG = 0x08;
	const FETCH_CATEGORY    = 0x10;
	const FETCH_RATING  = 0x20;
		
	/**
	 * Gets a single comment record specified by its ID
	 *
	 * @param integer $mediaId
	 *
	 * @return array
	 */
	public function getCommentById($commentId, array $fetchOptions = array())
	{
		$joinOptions = $this->prepareCommentFetchOptions($fetchOptions);

		return $this->_getDb()->fetchRow('
			SELECT comment.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_comment AS comment
				' . $joinOptions['joinTables'] . '
			WHERE comment.comment_id = ?
		', $commentId);
	}
	
	public function getCommentByRatingId($ratingId)
	{
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xengallery_comment
			WHERE rating_id = ?
		', $ratingId);
	}
	
	/**
	 * Gets comments based on options and criteria
	 *
	 * @param array $conditions
	 * @param array $fetchOptions
	 *
	 * @return array
	 */
	public function getComments(array $conditions = array(), array $fetchOptions = array())
	{
		$whereClause = $this->prepareCommentConditions($conditions, $fetchOptions);
		
		$joinOptions = $this->prepareCommentFetchOptions($fetchOptions);
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		$sqlClauses = $this->prepareCommentFetchOptions($fetchOptions);

		$comments = $this->fetchAllKeyed($this->limitQueryResults('
			SELECT comment.*
				' . $joinOptions['selectFields'] . '
			FROM xengallery_comment AS comment
				' . $joinOptions['joinTables'] . '
				WHERE ' . $whereClause . '
				' . $sqlClauses['orderClause'] . '
			', $limitOptions['limit'], $limitOptions['offset']
		), 'comment_id');
		
		return $comments;
	}
	
	public function countComments(array $conditions = array())
	{	
		$fetchOptions = array();
		$whereClause = $this->prepareCommentConditions($conditions, $fetchOptions);

		$joinOptions = $this->prepareCommentFetchOptions($fetchOptions);
				
		return $this->_getDb()->fetchOne('
			SELECT COUNT(*)
			FROM xengallery_comment AS comment
			' . $joinOptions['joinTables'] . '
			WHERE ' . $whereClause
		);
	}
	
	/**
	 * Fetches the most recent comments, together with user info.
	 *
	 * @param integer $maxComments If specified, limits the result. Otherwise fetches all comments.
	 *
	 * @return array
	 */
	public function getLatestComments($fetchOptions)
	{
		$limitOptions = $this->prepareLimitFetchOptions($fetchOptions);
		
		return $this->fetchAllKeyed(
			$this->limitQueryResults("
				SELECT
					comment.*,
					user.*
				FROM xengallery_comment AS comment
				LEFT JOIN xf_user AS user ON
					(user.user_id = comment.user_id)
				WHERE comment.media_id = $fetchOptions[media_id]
				ORDER BY comment_date DESC
				", $limitOptions['limit'], $limitOptions['offset']
			), 'comment_id'
		);
	}

	/**
	 * Fetches all comments together with user info.
	 *
	 * @return array
	 */
	public function getAllComments()
	{
		// just use getLatestComments with $maxComments = 0
		return $this->getLatestComments(0);
	}

	/**
	 * Fetches all comments newer than the date specified.
	 *
	 * @param integer Unix timestamp
	 *
	 * @return array
	 */
	public function getCommentsNewerThan($date, $mediaId)
	{
		return $this->fetchAllKeyed("
			SELECT
				comment.*,
				user.*
			FROM xengallery_comment AS comment
			LEFT JOIN xf_user AS user ON
				(user.user_id = comment.user_id)
			WHERE comment.comment_date > $date
			AND comment.media_id = $mediaId
			ORDER BY comment_date DESC
		", 'comment_id');
	}
	
	/**
	* Gets the most recent date
	*/
	public function getLatestDate()
	{
		return $this->_db->fetchOne('
			SELECT
				comment_date
			FROM xengallery_comment
			ORDER BY comment_date DESC
			LIMIT 1');
	}
	
	/**
	 * Attempts to update any instances of an old username in like_users with a new username
	 *
	 * @param integer $oldUserId
	 * @param integer $newUserId
	 * @param string $oldUsername
	 * @param string $newUsername
	 */
	public function batchUpdateLikeUser($oldUserId, $newUserId, $oldUsername, $newUsername)
	{
		$db = $this->_getDb();
	
		// note that xf_liked_content should have already been updated with $newUserId
	
		$db->query('
			UPDATE (
				SELECT content_id FROM xf_liked_content
				WHERE content_type = \'xengallery_comment\'
				AND like_user_id = ?
			) AS temp
			INNER JOIN xengallery_comment AS comment ON (comment.comment_id = temp.content_id)
			SET like_users = REPLACE(like_users, ' .
			$db->quote('i:' . $oldUserId . ';s:8:"username";s:' . strlen($oldUsername) . ':"' . $oldUsername . '";') . ', ' .
			$db->quote('i:' . $newUserId . ';s:8:"username";s:' . strlen($newUsername) . ':"' . $newUsername . '";') . ')
		', $newUserId);
	}
	
	/**
	* Given an attachment ID, delete all associated comments
	*/
	public function deleteCommentsByMediaId($mediaId)
	{
		return $this->_getDb()->query('
			DELETE
			FROM xengallery_comment
			WHERE media_id = ?
		', $mediaId);
	}
	
	/**
	* Checks that the viewing user may managed a reported media comment item
	*
	* @param array $comment
	* @param string $errorPhraseKey
	* @param array $viewingUser
	*
	* @return boolean
	*/
	public function canManageReportedComment(array $comment, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if (!$viewingUser['user_id'])
		{
			return false;
		}

		return (
			XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteCommentAny')
			|| XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCommentAny')
		);
	}
	
	public function canLikeComment(array $comment, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($comment['user_id'] == $viewingUser['user_id'])
		{
			$errorPhraseKey = 'xengallery_you_cannot_like_your_own_comment';
			return false;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'likeComment'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_like_permission';
		return false;		
	}	
	
	public function canEditComment(array $comment, &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($comment['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editComment'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'editCommentAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_edit_comment_permission';
		return false;		
	}
	
	public function canDeleteComment(array $comment, $type = 'soft', &$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);
		
		if ($comment['user_id'] == $viewingUser['user_id'] && XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteComment'))
		{
			return true;
		}

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'deleteCommentAny'))
		{
			return true;
		}
		
		$errorPhraseKey = 'xengallery_no_delete_comment_permission';
		return false;		
	}			
	
	public function canViewComments(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewComments'))
		{
			return true;
		}
	
		return false;
	}			
	
	public function canAddComment(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'addComment'))
		{
			return true;
		}
	
		return false;
	}			
	
	public function canViewDeletedComment(&$errorPhraseKey = '', array $viewingUser = null)
	{
		$this->standardizeViewingUserReference($viewingUser);

		if (XenForo_Permission::hasPermission($viewingUser['permissions'], 'xengallery', 'viewDeletedComments'))
		{
			return true;
		}
	
		return false;		
	}		
	
	public function prepareComments($comments)
	{
		$likeModel = $this->_getLikeModel();
		
		$visitor = XenForo_Visitor::getInstance();
		
		if (empty($comments['comment_id']))
		{
			foreach ($comments AS &$comment)
			{	
				$comment['canEdit'] = $this->canEditComment($comment);
				$comment['canDelete'] = $this->canDeleteComment($comment);
				$comment['canLike'] = $this->canLikeComment($comment);
				
				$comment['likeUsers'] = isset($comment['like_users']) ? unserialize($comment['like_users']) : false;
				
				$userIds = array();
				if (!empty($comment['likeUsers']))
				{
					foreach ($comment['likeUsers'] AS $user)
					{
						$userIds[$user['user_id']] = $user['user_id'];
					}					
				}
				
				$comment['existingLike'] = array_key_exists($visitor->user_id, $userIds);
				
				$comment['isIgnored'] = array_key_exists($comment['user_id'], $visitor->ignoredUsers);
				
				$comment['liked'] = ($comment['existingLike'] ? true : false);
				$comment['like_date'] = ($comment['liked'] ? XenForo_Application::$time : 0);
			}			
		}
		else
		{
			$comments['canEdit'] = $this->canEditComment($comments);
			$comments['canDelete'] = $this->canDeleteComment($comments);
			$comments['canLike'] = $this->canLikeComment($comments);
			
			$comments['likeUsers'] = isset($comments['like_users']) ? unserialize($comments['like_users']) : false;
			
			$userIds = array();
			if (!empty($comments['likeUsers']))
			{
				foreach ($comments['likeUsers'] AS $user)
				{
					$userIds[$user['user_id']] = $user['user_id'];
				}					
			}
			
			$comments['existingLike'] = array_key_exists($visitor->user_id, $userIds);
			
			$comments['isIgnored'] = array_key_exists($comments['user_id'], $visitor->ignoredUsers);
			
			$comments['liked'] = ($comments['existingLike'] ? true : false);
			$comments['like_date'] = ($comments['liked'] ? XenForo_Application::$time : 0);			
		}
		
		return $comments;
	}
	
	/**
	 * Prepares a set of conditions against which to select media comments.
	 *
	 * @param array $conditions List of conditions.
	 * @param array $fetchOptions The fetch options that have been provided. May be edited if criteria requires.
	 *
	 * @return string Criteria as SQL for where clause
	 */
	public function prepareCommentConditions(array $conditions, array &$fetchOptions)
	{
		$db = $this->_getDb();
		$sqlConditions = array();

		if (!empty($conditions['user_id']))
		{
			if (is_array($conditions['user_id']))
			{
				$sqlConditions[] = 'comment.user_id IN (' . $db->quote($conditions['user_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'comment.user_id = ' . $db->quote($conditions['user_id']);
			}
		}

		if (!empty($conditions['media_id']))
		{
			if (is_array($conditions['media_id']))
			{
				$sqlConditions[] = 'comment.media_id IN (' . $db->quote($conditions['media_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'comment.media_id = ' . $db->quote($conditions['media_id']);
			}
		}
		
		if (!empty($conditions['comment_id']))
		{
			if (is_array($conditions['comment_id']))
			{
				$sqlConditions[] = 'comment.comment_id IN (' . $db->quote($conditions['comment_id']) . ')';
			}
			else
			{
				$sqlConditions[] = 'comment.comment_id = ' . $db->quote($conditions['comment_id']);
			}
		}		

		if (isset($conditions['deleted']))
		{
			$sqlConditions[] = $this->prepareStateLimitFromConditions($conditions, 'comment', 'comment_state');
		}
		else
		{
			// sanity check: only get visible media unless we've explicitly said to get something else
			$sqlConditions[] = "comment.comment_state = 'visible'";
		}

		return $this->getConditionsForClause($sqlConditions);
	}

	/**
	 * Prepares join-related fetch options.
	 *
	 * @param array $fetchOptions
	 *
	 * @return array Containing 'selectFields' and 'joinTables' keys.
	 */	
	public function prepareCommentFetchOptions(array $fetchOptions)
	{
				
		$selectFields = '';
		$joinTables = '';		
		$orderBy = '';
		
		if (!empty($fetchOptions['order']))
		{
			$orderBySecondary = '';

			switch ($fetchOptions['order'])
			{
				case 'comment_date':
				default:
					$orderBy = 'comment.comment_date';
			}
			if (!isset($fetchOptions['orderDirection']) || $fetchOptions['orderDirection'] == 'ASC')
			{
				$orderBy .= ' ASC';
			}
			else
			{
				$orderBy .= ' DESC';
			}
		}

		if (!empty($fetchOptions['join']))
		{
			if ($fetchOptions['join'] & self::FETCH_MEDIA)
			{
				$selectFields .= ',
					media.media_id, media.media_title, media.media_description, media.media_state,
					media.comment_count, media.user_id, media.username, media.media_date';
				$joinTables .= '
					LEFT JOIN xengallery_media AS media ON
						(media.media_id = comment.media_id)';
			}

			if ($fetchOptions['join'] & self::FETCH_USER)
			{
				$selectFields .= ',
					user.*, user_profile.*, IF(user.username IS NULL, media.username, user.username) AS username';
				$joinTables .= '
					LEFT JOIN xf_user AS user ON
						(user.user_id = comment.user_id)
					LEFT JOIN xf_user_profile AS user_profile ON
						(user_profile.user_id = comment.user_id)';
			}

			if ($fetchOptions['join'] & self::FETCH_ATTACHMENT)
			{
				$selectFields .= ',
					attachment.attachment_id, attachment.data_id,
					attachment.view_count, attachment.attach_date,
					attachment_data.filename, attachment_data.file_size,
					attachment_data.thumbnail_width, attachment_data.file_hash';
				$joinTables .= '
					LEFT JOIN xf_attachment AS attachment ON
						(attachment.content_type = \'xengallery\' AND attachment.attachment_id = media.attachment_id)
					LEFT JOIN xf_attachment_data AS attachment_data ON
						(attachment_data.data_id = attachment.data_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_DELETION_LOG)
			{
				$selectFields .= ',
					deletion_log.delete_date, deletion_log.delete_reason,
					deletion_log.delete_user_id, deletion_log.delete_username';
				$joinTables .= '
					LEFT JOIN xf_deletion_log AS deletion_log ON
						(deletion_log.content_type = \'xengallery_comment\' AND deletion_log.content_id = comment.comment_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_CATEGORY)
			{
				$selectFields .= ',
					category.*';
				$joinTables .= '
					LEFT JOIN xengallery_category AS category ON
						(category.xengallery_category_id = media.xengallery_category_id)';
			}
			
			if ($fetchOptions['join'] & self::FETCH_RATING)
			{
				$selectFields .= ',
					rating.media_id, rating.media_rating_id, rating.rating, rating.rating_date';
				$joinTables .= '
					LEFT JOIN xengallery_rating AS rating ON
						(rating.media_rating_id = comment.rating_id)';
			}
		}

		return array(
			'selectFields' => $selectFields,
			'joinTables'   => $joinTables,
			'orderClause' => ($orderBy ? "ORDER BY $orderBy" : '')
		);
	}	
	
	protected function _getLikeModel()
	{
		return $this->getModelFromCache('XenForo_Model_Like');
	}
}