<?php
						
class Waindigo_Friends_Model_User extends XFCP_Waindigo_Friends_Model_User
{
	/**
	 * Creates a new friend request record for $userId friending $friendUserId
	 *
	 * @param array $friendUser User being friended
	 * @param array $user User doing the friending
	 * @param string $message
	 *
	 * @return string Comma-separated list of all users now being followed by $userId
	 */
	public function friend(array $friendUser, array $user = null, $message = '', $knowPersonally = false)
	{
		if ($user === null)
		{
			$user = XenForo_Visitor::getInstance();
		}
				
		$record = $this->getFriendRecord($friendUser['user_id'], $user['user_id']);
		
		if (!$record || $record['friend_state'] == 'pending'
				|| ($record['friend_state'] == 'rejected' && $record['friend_user_id'] == $user['user_id']))
		{
			$dw = XenForo_DataWriter::create('Waindigo_Friends_DataWriter_Friend');
			if ($record)
			{
				$dw->setExistingData(array($user['user_id'], $friendUser['user_id']));
			}
			if ($dw->get('user_id') == $friendUser['user_id'] && $dw->get('friend_user_id') == $user['user_id'])
			{
				$dw->set('friend_state', 'confirmed');
			}
			else
			{
				$dw->set('user_id', $user['user_id']);
				$dw->set('friend_user_id', $friendUser['user_id']);
			}
			if ($message)
			{
				$dw->set('message', $message);
			}
			$dw->set('know_personally', $knowPersonally);
			$dw->save();
			
			if ($dw->get('friend_state') == 'confirmed' || $dw->get('friend_state') == 'pending')
			{
				$this->_getAlertModel()->alert($friendUser['user_id'], $user['user_id'], $user['username'], 
						'friend', $user['user_id'], $dw->get('friend_state'));
			}
		}
	}

	/**
	 * Deletes a friend record between $userId and $friendUserId
	 *
	 * @param int $friendUser User Id being unfriended
	 * @param int $user User Id doing the unfriending
	 *
	 * @return string Comma-separated list of all users now being followed by $userId
	 */
	public function unfriend($friendUserId, $userId = null)
	{
		if ($userId === null)
		{
			$userId = XenForo_Visitor::getUserId();
		}
	
		$record = $this->getFriendRecord($userId, $friendUserId);
	
		if ($record && $record['friend_state'] != 'rejected')
		{
			$dw = XenForo_DataWriter::create('Waindigo_Friends_DataWriter_Friend');
			$dw->setExistingData(array($userId, $friendUserId));
			if ($dw->get('user_id') == $userId && $dw->get('friend_user_id') == $friendUserId)
			{
				$dw->set('user_id', $friendUserId);
				$dw->set('friend_user_id', $userId);
			}
			$dw->set('friend_state', 'rejected');
			$dw->save();
		}
	}
	
	/**
	 * Fetches a single friend record.
	 *
	 * @param integer $frienduserId - first user
	 * @param integer $userId - second user
	 *
	 * @return array
	 */
	public function getFriendRecord($friendUserId, $userId = null)
	{
		if ($userId === null)
		{
			$userId = XenForo_Visitor::getUserId();
		}
		
		return $this->_getDb()->fetchRow('
			SELECT *
			FROM xf_friend
			WHERE (user_id = ? AND friend_user_id = ?)
			OR (user_id = ? AND friend_user_id = ?)

			', array($userId, $friendUserId, $friendUserId, $userId));
	}

	/**
	 * Fetches multiple friend records.
	 *
	 * @param array $frienduserIds - first user
	 * @param integer $userId - second user
	 *
	 * @return array
	 */
	public function getFriendRecords(array $friendUserIds, $userId = null)
	{
		if ($userId === null)
		{
			$userId = XenForo_Visitor::getUserId();
		}
	
		return $this->fetchAllKeyed('
				SELECT *, if (user_id = ?, friend_user_id, user_id) AS user_id
				FROM xf_friend
				WHERE (user_id = ? AND friend_user_id IN (' . $this->_getDb()->quote($friendUserIds) . '))
				OR (user_id IN (' . $this->_getDb()->quote($friendUserIds) . ') AND friend_user_id = ?)
	
				', 'user_id', array($userId, $userId, $userId));
	}

	/**
	 * Fetches multiple friend records.
	 *
	 * @param array $frienduserIds - first user
	 * @param integer $userId - second user
	 *
	 * @return array
	 */
	public function getUserFriends($userId = null)
	{
		if ($userId === null)
		{
			$userId = XenForo_Visitor::getUserId();
		}
		
		$visitorUserId = XenForo_Visitor::getUserId();
	
		return $this->fetchAllKeyed('
				SELECT friend.*, user.*, mutual_friend.friend_state AS mutual_friend_state
				FROM xf_friend AS friend
				LEFT JOIN xf_user AS user
					ON (((user.user_id = friend.user_id AND friend.user_id != ?)
						OR (user.user_id = friend.friend_user_id AND friend.friend_user_id != ?)) AND user.is_banned = 0)
				LEFT JOIN xf_friend AS mutual_friend
					ON ((user.user_id = mutual_friend.user_id AND mutual_friend.friend_user_id = ?)
						OR (user.user_id = mutual_friend.friend_user_id AND mutual_friend.user_id = ?))
				WHERE (friend.user_id = ? OR friend.friend_user_id = ?)
					AND friend.friend_state = \'confirmed\'
			', 'user_id', array($userId, $userId, $visitorUserId, $visitorUserId, $userId, $userId));
	}
	
	/**
	 * Gets an array of all users who are friends with the specified user
	 *
	 * @param integer|array $userId|$user
	 * @param integer $maxResults (0 = all)
	 * @param string $orderBy
	 *
	 * @return array
	 */
	public function getFriendsUserProfiles($userId, $maxResults = 0, $orderBy = 'user.username')
	{
		$sql = '
			SELECT
				user.*,
				user_profile.*,
				user_option.*
			FROM xf_friend AS friend
			LEFT JOIN xf_user AS user
				ON (((user.user_id = friend.user_id AND friend.user_id != ?)
					OR (user.user_id = friend.friend_user_id AND friend.friend_user_id != ?)) AND user.is_banned = 0)
			INNER JOIN xf_user_profile AS user_profile ON
				(user_profile.user_id = user.user_id)
			INNER JOIN xf_user_option AS user_option ON
				(user_option.user_id = user.user_id)
			WHERE (friend.user_id = ? OR friend.friend_user_id = ?)
				AND friend.friend_state = \'confirmed\'
			ORDER BY ' . $orderBy . '
		';

		if ($maxResults)
		{
			$sql = $this->limitQueryResults($sql, $maxResults);
		}

		$userId = $this->getUserIdFromUser($userId);
		return $this->fetchAllKeyed($sql, 'user_id', array($userId, $userId, $userId, $userId));
	}
	
	/**
	 * Gets an array of all users who are friends with the specified user
	 *
	 * @param integer|array $userId|$user
	 * @param integer $maxResults (0 = all)
	 * @param string $orderBy
	 *
	 * @return array
	 */
	public function getMutualFriendsUserProfiles($userId, $maxResults = 0, $orderBy = 'user.username')
	{
		$sql = '
			SELECT
				user.*,
				user_profile.*,
				user_option.*
			FROM xf_friend AS friend
			INNER JOIN xf_user AS user
				ON (((user.user_id = friend.user_id AND friend.user_id != ?)
					OR (user.user_id = friend.friend_user_id AND friend.friend_user_id != ?)) AND user.is_banned = 0)
			INNER JOIN xf_friend AS mutual_friend
				ON ((user.user_id = mutual_friend.user_id AND mutual_friend.friend_user_id = ?)
					OR (user.user_id = mutual_friend.friend_user_id AND mutual_friend.user_id = ?))
			INNER JOIN xf_user_profile AS user_profile ON
				(user_profile.user_id = user.user_id)
			INNER JOIN xf_user_option AS user_option ON
				(user_option.user_id = user.user_id)
			WHERE (friend.user_id = ? OR friend.friend_user_id = ?)
				AND friend.friend_state = \'confirmed\'
				AND mutual_friend.user_id IS NOT NULL
			ORDER BY ' . $orderBy . '
		';
	
		if ($maxResults)
		{
			$sql = $this->limitQueryResults($sql, $maxResults);
		}
	
		$userId = $this->getUserIdFromUser($userId);
		$visitorId = XenForo_Visitor::getUserId();
		return $this->fetchAllKeyed($sql, 'user_id', array($userId, $userId, $visitorId, $visitorId, $userId, $userId));
	}
	
	/**
	 * @return XenForo_Model_Alert
	 */
	protected function _getAlertModel()
	{
		return $this->getModelFromCache('XenForo_Model_Alert');
	}
}