<?php

/**
* Data writer for media comments
*/
class XenGallery_DataWriter_Comment extends XenForo_DataWriter
{
	/**
	 * Holds the reason for soft deletion.
	 *
	 * @var string
	 */
	const DATA_DELETE_REASON = 'deleteReason';
		
	/**
	 * Title of the phrase that will be created when a call to set the
	 * existing data fails (when the data doesn't exist).
	 *
	 * @var string
	 */
	protected $_existingDataErrorPhrase = 'xengallery_requested_comment_not_found';

	/**
	* Gets the fields that are defined for the table. See parent for explanation.
	*
	* @return array
	*/
	protected function _getFields()
	{
		return array(
			'xengallery_comment' => array(
				'comment_id' => array('type' => self::TYPE_UINT, 'autoIncrement' => true),
				'media_id' => array('type' => self::TYPE_UINT, 'required' => true),
				'media_comment' => array('type' => self::TYPE_STRING, 'required' => true,
					'requiredError' => 'please_enter_valid_message'
				),
				'user_id' => array('type' => self::TYPE_UINT, 'required' => true),
				'username' => array('type' => self::TYPE_STRING, 'required' => true, 'maxLength' => 50,
					'requiredError' => 'please_enter_valid_username'
				),
				'comment_date'  => array('type' => self::TYPE_UINT,   'required' => true, 'default' => XenForo_Application::$time),
				'comment_state' => array('type' => self::TYPE_STRING, 'default' => 'visible',
					'allowedValues' => array('visible', 'moderated', 'deleted')
				),
				'rating_id' => array('type' => self::TYPE_UINT, 'default' => 0),
                'likes'
                    => array('type' => self::TYPE_UINT),
                'like_users'
                    => array('type' => self::TYPE_SERIALIZED),				
			)
		);
	}

	/**
	* Gets the actual existing data out of data that was passed in. See parent for explanation.
	*
	* @param mixed
	*
	* @return array|false
	*/
	protected function _getExistingData($data)
	{
		if (!$id = $this->_getExistingPrimaryKey($data))
		{
			return false;
		}

		return array('xengallery_comment' => $this->_getCommentModel()->getCommentById($id));
	}

	/**
	* Gets SQL condition to update the existing record.
	*
	* @return string
	*/
	protected function _getUpdateCondition($tableName)
	{
		return 'comment_id = ' . $this->_db->quote($this->getExisting('comment_id'));
	}

	protected function _preSave()
	{
		
	}

	protected function _postSave()
	{
		if ($this->isInsert())
		{
			$mediaId = $this->get('media_id');
			
			$mediaWriter = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			$mediaWriter->setExistingData($mediaId);
			
			$commentCount = $mediaWriter->getExisting('comment_count');
			
			$mediaWriter->set('comment_count', $commentCount + 1);
			$mediaWriter->set('last_comment_date', XenForo_Application::$time);
			
			$media = $this->_getMediaModel()->getMediaById($mediaId, array(
				'join' => XenGallery_Model_Media::FETCH_USER | XenGallery_Model_Media::FETCH_USER_OPTION
			));
			
			$this->_getMediaModel()->markMediaViewed($media);			
			
			$commentUser = array(
				'user_id' => $this->get('user_id'),
				'username' => $this->get('username')			
			);			
			
			$commentId = $this->get('comment_id');
			
			if ($media && XenForo_Model_Alert::userReceivesAlert($media, 'xengallery_comment', 'insert') && $media['user_id'] != $commentUser['user_id'])
			{					
				XenForo_Model_Alert::alert(
					$media['user_id'],
					$commentUser['user_id'],
					$commentUser['username'],
					'xengallery_comment',
					$commentId,
					'insert'
				);			
			}
			
			$mediaWriter->save();
			
			$this->_getNewsFeedModel()->publish(
				$commentUser['user_id'],
				$commentUser['username'],
				'xengallery_comment',
				$commentId,
				'insert'
			);					
			
			$ipId = XenForo_Model_Ip::log(
				$this->get('user_id'), 'xengallery_comment', $this->get('comment_id'), 'insert'
			);
			
			$this->_db->update('xengallery_comment', array(
				'ip_id' => $ipId
			), 'comment_id = ' . $this->get('comment_id'));				
		}
		
		if ($this->isChanged('comment_state') && $this->getExisting('comment_state') == 'deleted')
		{
			$this->updateCommentCount();
		}
		
		if ($this->isChanged('comment_state') && $this->getExisting('comment_state') == 'visible')
		{
			$this->updateCommentCount(false);
		}		
		
		$this->_updateDeletionLog(true);
	}

	/**
	 * Post-delete handling.
	 */
	protected function _postDelete()
	{
		$this->_updateDeletionLog(true);
		$this->updateCommentCount(false);
	}
	
	
	public function updateCommentCount($increase = true)
	{
		$value = '+ 1';
		
		if (!$increase)
		{
			$value = '- 1';
		}
		
		$this->_db->query("
			UPDATE xengallery_media
			SET comment_count = comment_count $value
			WHERE media_id = ?
		", $this->get('media_id'));
	}	
	
	protected function _updateDeletionLog($hardDelete = false)
	{
		if ($hardDelete
			|| ($this->isChanged('comment_state') && $this->getExisting('comment_state') == 'deleted')
		)
		{
			$this->getModelFromCache('XenForo_Model_DeletionLog')->removeDeletionLog(
				'xengallery_comment', $this->get('comment_id')
			);			
		}

		if ($this->isChanged('comment_state') && $this->get('comment_state') == 'deleted')
		{
			$reason = $this->getExtraData(self::DATA_DELETE_REASON);
			$this->getModelFromCache('XenForo_Model_DeletionLog')->logDeletion(
				'xengallery_comment', $this->get('comment_id'), $reason
			);
		}
	}		

	/**
	 * @return XenGallery_Model_Comment
	 */
	protected function _getCommentModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Comment');
	}
	
	/**
	 * @return XenGallery_Model_Media
	 */
	protected function _getMediaModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Media');
	}	
}