<?php
/**
 * @package PowerDigital_Unanswered_Controller
 * @author Luqman Amjad http://luqmanamjad.com
 */
class PowerDigital_Unanswered_Controller extends XenForo_ControllerPublic_Abstract
{
  const SEARCH_TYPE = 'unanswered-threads';
  
  /**
   * @var XenForo_Options Object
   */
  private static $options = null;
  
  /**
   * Setup options for each action
   */
  public function _preDispatch($action)
  {
    self::$options = XenForo_Application::get('options');
  }
  
  /**
   * Finds unanswered threads
   *
   * @return XenForo_ControllerResponse_Abstract
   */
  public function actionThreads()
  {
    $searchId = $this->_input->filterSingle('search_id', XenForo_Input::UINT);
    if (!$searchId)
    {
      return $this->findUnansweredReplies();
    }
    
    $searchModel = $this->_getSearchModel();
    
    $search = $searchModel->getSearchById($searchId);
    if (!$search || $search['user_id'] != XenForo_Visitor::getUserId() || !in_array($search['search_type'], array(self::SEARCH_TYPE)))
    {
      return $this->findUnansweredReplies();
    }
    
    $page = max(1, $this->_input->filterSingle('page', XenForo_Input::UINT));
    $perPage = self::$options->discussionsPerPage;
    
    $pageResultIds = $searchModel->sliceSearchResultsToPage($search, $page, $perPage);
    $results = $searchModel->getSearchResultsForDisplay($pageResultIds);
    if (!$results)
    {
      return $this->getNoResultsResponse();
    }
    
    $resultStartOffset = ($page - 1) * $perPage + 1;
    $resultEndOffset = ($page - 1) * $perPage + count($results['results']);
    
    $threadModel = $this->_getThreadModel();
    
    $threads = array();
    $inlineModOptions = array();
    foreach ($results['results'] AS $result)
    {
      $thread = $result['content'];
      
      $thread['forum'] = array(
        'node_id' => $thread['node_id'],
        'title' => $thread['node_title']
      );
      
      $threadModOptions = $threadModel->addInlineModOptionToThread($thread, $thread, $thread['permissions']);
      $inlineModOptions += $threadModOptions;
      
      $threads[$result[XenForo_Model_Search::CONTENT_ID]] = $thread;
    }
    
    $viewParams = array(
      'search' => $search,
      'threads' => $threads,
      'inlineModOptions' => $inlineModOptions,
      
      'threadStartOffset' => $resultStartOffset,
      'threadEndOffset' => $resultEndOffset,
      
      'page' => $page,
      'perPage' => $perPage,
      'totalThreads' => $search['result_count'],
      'nextPage' => ($resultEndOffset < $search['result_count'] ? ($page + 1) : 0),
      'alternativeDaysToLookBack' => self::$options->alternativeDaysToLookBack
    );
    
    return $this->responseView('PowerDigital_Unanswered_ViewPublic_Unanswered_Threads', 'unanswered_threads', $viewParams);
  }

  public function findUnansweredReplies()
  {
    $threadModel = $this->_getThreadModel();
    $searchModel = $this->_getSearchModel();
    
    $userId = XenForo_Visitor::getUserId();
    
    $limitOptions = array(
      'limit' => self::$options->maximumSearchResults
    );
    
    $days = $this->_input->filterSingle('days', XenForo_Input::UINT);
    $forum_id = $this->_input->filterSingle('forum_id', XenForo_Input::STRING);
    
    if ($days < 1)
    {
      $days = self::$options->miniumDaysToLookBack;
    }
    
    $fetchOptions = $limitOptions + array(
      'order' => 'last_post_date',
      'orderDirection' => 'desc',
    );
    
    $threadIds = array_keys($threadModel->getUnansweredThreads(array(
      'last_post_date' => array('>', XenForo_Application::$time - 86400 * $days),
      'deleted' => false,
      'moderated' => false,
      'forum_id' => $forum_id
    ), $fetchOptions));
    
    $searchType = self::SEARCH_TYPE;
    
    $results = array();
    foreach ($threadIds AS $threadId)
    {
      $results[] = array(
        XenForo_Model_Search::CONTENT_TYPE => 'thread',
        XenForo_Model_Search::CONTENT_ID => $threadId);
    }
    
    $results = $searchModel->getViewableSearchResults($results);
    if (!$results)
    {
      return $this->getNoResultsResponse();
    }
    
    $search = $searchModel->insertSearch($results, $searchType, '', array(), 'date', false);
    
    return $this->responseRedirect(
      XenForo_ControllerResponse_Redirect::SUCCESS,
      XenForo_Link::buildPublicLink('unanswered/threads', $search)
    );
  }

  public function getNoResultsResponse()
  {
    $days = $this->_input->filterSingle('days', XenForo_Input::UINT);
    
    if (XenForo_Visitor::getUserId() && !$days)
    {
      return $this->responseMessage(new XenForo_Phrase('no_unanswered_threads_found',
        array(
          'days' => self::$options->miniumDaysToLookBack, 'alternative_days' => self::$options->alternativeDaysToLookBack,
          'link' => XenForo_Link::buildPublicLink('unanswered/threads', false, array('days' => self::$options->alternativeDaysToLookBack))
        )
      ));
    }
    else
    {
      return $this->responseMessage(new XenForo_Phrase('no_results_found'));
    }
  }

  /**
   * Session activity details.
   * @see XenForo_Controller::getSessionActivityDetailsForList()
   */
  public static function getSessionActivityDetailsForList(array $activities)
  {
    return new XenForo_Phrase('viewing_unanswered_content');
  }

  /**
   * @return XenForo_Model_Thread
   */
  protected function _getThreadModel()
  {
    return $this->getModelFromCache('XenForo_Model_Thread');
  }

  /**
   * @return XenForo_Model_Search
   */
  protected function _getSearchModel()
  {
    return $this->getModelFromCache('XenForo_Model_Search');
  }
}