<?php

/**
 * Controller for resource categories
 */
class XenResource_ControllerAdmin_Category extends XenForo_ControllerAdmin_Abstract
{
	protected function _preDispatch($action)
	{
		$this->assertAdminPermission('resourceManager');
	}

	/**
	 * Lists all categories.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionIndex()
	{
		$viewParams = array(
			'categories' => $this->_getCategoryModel()->getAllCategories()
		);
		return $this->responseView('XenResource_ViewAdmin_Category_List', 'resource_category_list', $viewParams);
	}

	/**
	 * Gets the category add/edit form response.
	 *
	 * @param array $category
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	protected function _getCategoryAddEditResponse(array $category)
	{
		if (!empty($category['resource_category_id']))
		{
			$categories = $this->_getCategoryModel()->getPossibleParentCategories($category);
		}
		else
		{
			$categories = $this->_getCategoryModel()->getAllCategories();
		}

		if (!empty($category['thread_node_id']))
		{
			$prefixes = $this->getModelFromCache('XenForo_Model_ThreadPrefix')->getPrefixOptions(array(
				'node_id' => $category['thread_node_id']
			));
		}
		else
		{
			$prefixes = array();
		}

		$userGroups = $this->getModelFromCache('XenForo_Model_UserGroup')->getAllUserGroups();

		$selUserGroupIds = explode(',', $category['allow_submit_user_group_ids']);
		if (in_array(-1, $selUserGroupIds))
		{
			$allUserGroups = true;
			$selUserGroupIds = array_keys($userGroups);
		}
		else
		{
			$allUserGroups = false;
		}

		$viewParams = array(
			'category' => $category,
			'categories' => $categories,
			'nodes' => $this->getModelFromCache('XenForo_Model_Node')->getAllNodes(),
			'prefixes' => $prefixes,

			'allUserGroups' => $allUserGroups,
			'selUserGroupIds' => $selUserGroupIds,
			'userGroups' => $userGroups
		);
		return $this->responseView('XenResource_ViewAdmin_Category_Edit', 'resource_category_edit', $viewParams);
	}

	/**
	 * Displays a form to create a new category.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionAdd()
	{
		return $this->_getCategoryAddEditResponse(array(
			'display_order' => 1,
			'allow_local' => 1,
			'allow_external' => 1,
			'allow_commercial_external' => 1,
			'allow_fileless' => 1,
			'allow_submit_user_group_ids' => '-1'
		));
	}

	/**
	 * Displays a form to edit an existing category.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionEdit()
	{
		$category = $this->_getCategoryOrError();

		return $this->_getCategoryAddEditResponse($category);
	}

	/**
	 * Updates an existing media site or inserts a new one.
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionSave()
	{
		$this->_assertPostOnly();

		$categoryId = $this->_input->filterSingle('resource_category_id', XenForo_Input::STRING);

		$dwInput = $this->_input->filter(array(
			'category_title' => XenForo_Input::STRING,
			'category_description' => XenForo_Input::STRING,
			'parent_category_id' => XenForo_Input::UINT,
			'display_order' => XenForo_Input::UINT,
			'allow_local' => XenForo_Input::UINT,
			'allow_external' => XenForo_Input::UINT,
			'allow_commercial_external' => XenForo_Input::UINT,
			'allow_fileless' => XenForo_Input::UINT,
			'thread_node_id' => XenForo_Input::UINT,
			'thread_prefix_id' => XenForo_Input::UINT,
			'always_moderate_create' => XenForo_Input::UINT,
			'always_moderate_update' => XenForo_Input::UINT,
		));

		$input = $this->_input->filter(array(
			'usable_user_group_type' => XenForo_Input::STRING,
			'user_group_ids' => array(XenForo_Input::UINT, 'array' => true)
		));

		if ($input['usable_user_group_type'] == 'all')
		{
			$allowedGroupIds = array(-1); // -1 is a sentinel for all groups
		}
		else
		{
			$allowedGroupIds = $input['user_group_ids'];
		}

		$dw = XenForo_DataWriter::create('XenResource_DataWriter_Category');
		if ($categoryId)
		{
			$dw->setExistingData($categoryId);
		}
		$dw->bulkSet($dwInput);
		$dw->set('allow_submit_user_group_ids', $allowedGroupIds);
		$dw->save();

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildAdminLink('resource-categories') . $this->getLastHash($dw->get('resource_category_id'))
		);
	}

	/**
	 * Deletes the specified category
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	public function actionDelete()
	{
		if ($this->isConfirmedPost())
		{
			return $this->_deleteData(
				'XenResource_DataWriter_Category', 'resource_category_id',
				XenForo_Link::buildAdminLink('resource-categories/delete-clean-up', null, array(
					'resource_category_id' => $this->_input->filterSingle('resource_category_id', XenForo_Input::UINT),
					'_xfToken' => XenForo_Visitor::getInstance()->csrf_token_page
				))
			);
		}
		else // show confirmation dialog
		{
			$viewParams = array(
				'category' => $this->_getCategoryOrError()
			);
			return $this->responseView('XenResource_ViewAdmin_Category_Delete', 'resource_category_delete', $viewParams);
		}
	}

	public function actionDeleteCleanUp()
	{
		$this->_checkCsrfFromToken($this->_input->filterSingle('_xfToken', XenForo_Input::STRING));

		$id = $this->_input->filterSingle('resource_category_id', XenForo_Input::UINT);

		$info = $this->_getCategoryModel()->getCategoryById($id);
		if (!$id || $info)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL,
				XenForo_Link::buildAdminLink('resource-categories')
			);
		}

		$resources = $this->_getResourceModel()->getResources(array('resource_category_id' => $id), array('limit' => 100));
		if (!$resources)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL,
				XenForo_Link::buildAdminLink('resource-categories')
			);
		}

		$start = microtime(true);
		$limit = 10;

		foreach ($resources AS $resource)
		{
			$dw = XenForo_DataWriter::create('XenResource_DataWriter_Resource', XenForo_DataWriter::ERROR_SILENT);
			$dw->setExistingData($resource);
			$dw->delete();

			if ($limit && microtime(true) - $start > $limit)
			{
				break;
			}
		}

		return $this->responseView('XenResource_ViewAdmin_Category_DeleteCleanUp', 'resource_category_delete_clean_up', array(
			'resource_category_id' => $id
		));
	}

	/**
	 * Gets the specified record or errors.
	 *
	 * @param string $id
	 *
	 * @return array
	 */
	protected function _getCategoryOrError($id = null)
	{
		if ($id === null)
		{
			$id = $this->_input->filterSingle('resource_category_id', XenForo_Input::UINT);
		}

		$info = $this->_getCategoryModel()->getCategoryById($id);
		if (!$info)
		{
			throw $this->responseException($this->responseError(new XenForo_Phrase('requested_category_not_found'), 404));
		}

		return $info;
	}

	/**
	 * @return XenResource_Model_Resource
	 */
	protected function _getResourceModel()
	{
		return $this->getModelFromCache('XenResource_Model_Resource');
	}

	/**
	 * @return XenResource_Model_Category
	 */
	protected function _getCategoryModel()
	{
		return $this->getModelFromCache('XenResource_Model_Category');
	}
}