<?php

class XenGallery_ControllerPublic_Media extends XenForo_ControllerPublic_Abstract
{
	/**
	 * Enforce view permissions for all actions in this controller
	 *
	 * @see library/XenForo/XenForo_Controller#_preDispatch($action)
	 */
	protected function _preDispatch($action)
	{
		if (!$this->_getMediaModel()->canViewMedia($error))
		{
			throw $this->getErrorOrNoPermissionResponseException($error);
		}
	}
	
	public function actionIndex()
	{
		$this->_routeMatch->setSections('xengallery', '');
		
		$mediaModel = $this->_getMediaModel();
		
		$order = $this->_input->filterSingle('order', XenForo_Input::STRING);
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		if ($mediaId)
		{
			$media = $mediaModel->getMediaById($mediaId);
			
			return $this->responseReroute(__CLASS__, 'view');
		}
		
		$visitor = XenForo_Visitor::getInstance();
		
		$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
		$perPage = XenForo_Application::get('options')->xengalleryMediaMaxPerPageHome;
		
		$conditions = array(
			'deleted' => XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'xengallery', 'viewDeleted')
		);	
		$fetchOptions = $this->_getMediaFetchOptions() + array(
			'order' => $order ? $order : 'media_date',
			'orderDirection' => 'desc',
			'page' => $page,
			'perPage' => $perPage
		);
		
		$media = $mediaModel->getMedia($conditions, $fetchOptions);
		$media = $mediaModel->prepareMediaItems($media);
		
		$ignoredNames = array();
		foreach ($media AS $item)
		{
			if (isset($item['isIgnored']))
			{
				$ignoredNames[] = $item['username'];
			}
		}		
		
		$mediaCount = $mediaModel->countMedia($conditions);
		
		$definedPrefix = XenForo_Application::get('options')->xengalleryRoutePrefix;
		
		$this->canonicalizePageNumber($page, $perPage, $mediaCount, $definedPrefix);
		
		$viewParams = array(
			'canAddMedia' => $mediaModel->canAddMedia(),
			'media' => $media,
			'ignoredNames' => array_unique($ignoredNames),			
			'mediaCount' => $mediaCount,
			'page' => $page,
			'perPage' => $perPage,
			'order' => $order,
			'time' => XenForo_Application::$time
		);
		
		return $this->_getWrapper('',
			$this->responseView('XenGallery_ViewPublic_Media_HomeCategory', 'xengallery_home', $viewParams)
		);
	}
	
	public function actionMarkViewed()
	{		
		$requestParams = $this->getRequest()->getParams();
		
		$categoryId = $this->_input->filterSingle('category_id', XenForo_Input::UINT);
		if (!empty($requestParams['_xfRequestUri']))
		{
			if (strstr($requestParams['_xfRequestUri'], 'categories'))
			{
				$dotPos = strpos($requestParams['_xfRequestUri'], '.');
				$categoryId = (int)preg_replace("/[^0-9]/", '', substr($requestParams['_xfRequestUri'], $dotPos));
			}
		}
		
		$category = $this->_getCategoryModel()->getCategoryById($categoryId);
		
		$categoryOnly = $this->_input->filterSingle('category_only', XenForo_Input::STRING);
		
		if ($this->isConfirmedPost())
		{
			$mediaModel = $this->_getMediaModel();
			
			$fetchOptions = array();
			if ($categoryOnly)
			{
				$fetchOptions['xengallery_category_id'] = $categoryId;
			}
			
			$mediaIds = $mediaModel->getUnviewedMediaIds(XenForo_Visitor::getInstance()->getUserId(), $fetchOptions);
			
			$db = XenForo_Application::getDb();			
			XenForo_Db::beginTransaction($db);						
			
			if (sizeof($mediaIds))
			{
				$media = $mediaModel->getMediaByIds($mediaIds);
				
				foreach ($media AS $item)
				{
					$mediaModel->markMediaViewed($item);
				}				
			}
			
			XenForo_Db::commit($db);
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				$this->getDynamicRedirect()
			);
		}
		else
		{
			$viewParams = array(
				'markDate' => XenForo_Application::$time,
				'category' => $category
			);
			
			return $this->responseView('XenGallery_ViewPublic_Media_MarkViewed', 'xengallery_mark_viewed', $viewParams);
		}
	}
	
	public function actionView()
	{		
		$mediaModel = $this->_getMediaModel();
		$attachmentModel = $this->_getAttachmentModel();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		
		$fetchOptions = $this->_getMediaFetchOptions();
		$fetchOptions['join'] |= XenGallery_Model_Media::FETCH_NEXT_PREV;
		
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId, $fetchOptions);
		
		$mediaModel->markMediaViewed($media);
		$mediaModel->logMediaView($media['media_id']);
		
		$likeModel = $this->_getLikeModel();
		$existingLike = $likeModel->getContentLikeByLikeUser('xengallery_media', $mediaId, XenForo_Visitor::getUserId());

		$liked = ($existingLike ? true : false);
		
		$media['likeUsers'] = isset($media['like_users']) ? unserialize($media['like_users']) : false;
		$media['like_date'] = ($liked ? XenForo_Application::$time : 0);
		
		$commentModel = $this->_getCommentModel();
		
		$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
		$perPage = XenForo_Application::get('options')->xengalleryMaxMediaCommentsPerPage;
		
		$conditions = array(
			'media_id' => $mediaId,
			'deleted' => XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'xengallery', 'viewDeletedComments')
		);
		
		$fetchOptions = $this->_getCommentFetchOptions() + array(
			'page' => $page,
			'perPage' => $perPage,
			'order' => 'comment_date',
			'orderDirection' => 'desc'
		);
		
		$comments = $commentModel->getComments($conditions, $fetchOptions);
		$comments = $commentModel->prepareComments($comments);
		
		$ignoredNames = array();
		foreach ($comments AS $comment)
		{
			if (isset($comment['isIgnored']))
			{
				$ignoredNames[] = $comment['username'];
			}
		}
		
		$ignoredNames = array_unique($ignoredNames);
		
		$commentCount = $commentModel->countComments($conditions, $fetchOptions);
		
		$date = $comments ? $commentModel->getLatestDate() : 0;		
		
		$this->canonicalizePageNumber($page, $perPage, $commentCount, 'xenmediagallery', $media);
		$this->canonicalizeRequestUrl(
			XenForo_Link::buildPublicLink('xenmediagallery', $media, array('page' => $page))
		);
		
		$visitor = XenForo_Visitor::getInstance();
				
		$viewParams = array(
			'media' => $media,
			'next' => array('media_id' => $media['next_id'], 'media_title' => $media['next_title']),
			'prev' => array('media_id' => $media['prev_id'], 'media_title' => $media['prev_title']),
			'liked' => $liked,
			'canDeleteMedia' => $mediaModel->canDeleteMedia($media),
			'canEditMedia' => $mediaModel->canEditMedia($media),
			'canApproveUnapprove' => $mediaModel->canApproveMedia($media) || $mediaModel->canUnapproveMedia($media),
			'canLikeMedia' => $mediaModel->canLikeMedia($media),
			'canRateMedia' => $mediaModel->canRateMedia($media),
			'canReport' => $visitor['user_id'] ? true : false,
			'canEditCaption' => $mediaModel->canEditCaption($media),
			'canViewComments' => $commentModel->canViewComments(),
			'canAddComment' => $commentModel->canAddComment(),			
			'canViewIps' => XenForo_Permission::hasPermission($visitor->permissions, 'general', 'viewIps'),
			'comments' => $comments,			
			'commentCount' => $commentCount,
			'ignoredNames' => $ignoredNames,
			'page' => $page,
			'perPage' => $perPage,
			'date' => $date,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media, true),
			'externalDataUrl' => XenForo_Application::get('config')->externalDataUrl
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_View', 'xengallery_view', $viewParams);
	}
	
	public function actionRate()
	{
		$mediaModel = $this->_getMediaModel();
		$categoryModel = $this->_getCategoryModel();
		
		$mediaHelper = $this->_getMediaHelper();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $mediaHelper->assertMediaValidAndViewable($mediaId);
		
		$categoryBreadcrumbs = $categoryModel->getCategoryBreadcrumb($media);
		
		$mediaHelper->assertCanRateMedia($media);

		$visitor = XenForo_Visitor::getInstance();

		$input = $this->_input->filter(array(
			'rating' => XenForo_Input::UINT,
			'media_comment' => XenForo_Input::STRING			
		));

		$existing = $this->_getRatingModel()->getRatingByMediaAndUserId($mediaId, $visitor['user_id']);
		
		$comment = '';
		if ($existing)
		{
			$comment = $this->_getCommentModel()->getCommentByRatingId($existing['media_rating_id']);		
		}

		if ($this->isConfirmedPost())
		{
			$ratingDw = XenForo_DataWriter::create('XenGallery_DataWriter_Rating', XenForo_DataWriter::ERROR_EXCEPTION);
			$ratingDw->set('media_id', $media['media_id']);
			$ratingDw->set('user_id', $visitor['user_id']);
			$ratingDw->set('username', $visitor['username']);
			$ratingDw->set('rating', $input['rating']);

			if ($existing)
			{
				$deleteDw = XenForo_DataWriter::create('XenGallery_DataWriter_Rating');
				$deleteDw->setExistingData($existing, true);
				$deleteDw->delete();
			}
			
			$commentDw = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');
			
			if ($existing && $comment)
			{
				$commentDw->setExistingData($comment);
				
				if (!$input['media_comment'])
				{
					$commentDw->delete();
				}
			}

			$ratingDw->save();
			
			if ($input['media_comment'])
			{				
				$data = array(
					'media_id' => $mediaId,
					'media_comment' => $input['media_comment'],
					'user_id' => $visitor['user_id'],
					'username' => $visitor['username'],
					'rating_id' => $ratingDw->get('rating_id')
				);
				
				$commentDw->bulkSet($data);
				$commentDw->save();
			}

			$mediaDw = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			$newRating = $mediaModel->getRatingAverage($mediaDw->get('rating_sum'), $mediaDw->get('rating_count'), true);
			$hintText = new XenForo_Phrase('x_votes', array('count' => $mediaDw->get('rating_count')));

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $media),
				new XenForo_Phrase('your_rating_has_been_recorded'),
				array(
					'newRating' => $newRating,
					'hintText' => $hintText
				)
			);
		}
		else
		{
			$viewParams = array(
				'media' => $media,
				'rating' => $input['rating'],
				'existing' => ($existing ? $existing : false),

				'categoryBreadcrumbs' => $categoryBreadcrumbs,
			);
				
			if ($comment)
			{
				$viewParams += array(
					'media_comment' => $comment['media_comment']
				);					
			}

			return $this->responseView('XenGallery_ViewPublic_Media_Rate', 'xengallery_media_rate', $viewParams);
		}
	}	
	
	public function actionRatings()
	{
		$mediaHelper = $this->_getMediaHelper();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media =  $mediaHelper->assertMediaValidAndViewable($mediaId);
		
		$conditions = array(
			'media_id' => $mediaId
		);
		
		$fetchOptions = array(
			'join' => XenGallery_Model_Rating::FETCH_USER
		);
		
		$ratings = $this->_getRatingModel()->getRatings($conditions, $fetchOptions);
		
		$viewParams = array(
			'media' => $media,
			'ratings' => $ratings,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media)
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_Ratings', 'xengallery_media_ratings', $viewParams);			
	}
	
	
	public function actionLikes()
	{
		$mediaHelper = $this->_getMediaHelper();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media =  $mediaHelper->assertMediaValidAndViewable($mediaId);
		
		$conditions = array(
			'media_id' => $mediaId
		);
		
		$fetchOptions = array(
			'join' => XenGallery_Model_Rating::FETCH_USER
		);
		
		$likes = $this->_getLikeModel()->getContentLikes('xengallery_media', $mediaId);
		if (!$likes)
		{
			return $this->responseError(new XenForo_Phrase('xengallery_no_one_has_liked_this_media_yet'));
		}
					
		$viewParams = array(
			'media' => $media,
			'likes' => $likes,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media)
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_Ratings', 'xengallery_media_likes', $viewParams);		
	}
	
	public function actionPreview()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$mediaModel = $this->_getMediaModel();
		
		$media = $mediaModel->getMediaById($mediaId, $this->_getMediaFetchOptions());
		
		if ($this->_noRedirect())
		{
			$viewParams = array(
				'media' => $media
			);

			return $this->responseView('XenGallery_ViewPublic_Media_Preview', 'xengallery_media_preview', $viewParams);
		}
		else
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $media)
			);
		}		
	}
	
	public function actionFull()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
		
		if ($media['media_type'] == 'video_embed')
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery', $media)
			);
		}		
		
		$attachmentId = $media['attachment_id'];
        
        $this->_request->setParam('attachment_id', $media['attachment_id']);
            
        return $this->responseReroute('XenForo_ControllerPublic_Attachment', 'index');
	}
	
	public function actionDownload()
	{
		$attachmentModel = $this->_getAttachmentModel();
		$mediaModel = $this->_getMediaModel();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
		
		if ($media['media_type'] == 'video_embed')
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery', $media)
			);
		}
		
		$filePath = $attachmentModel->getAttachmentDataFilePath($media);
		if (!file_exists($filePath) || !is_readable($filePath))
		{
			return $this->responseError(new XenForo_Phrase('xengallery_media_cannot_be_downloaded'));
		}
		
		$this->_routeMatch->setResponseType('raw');
		
		$viewParams = array(
			'media' => $media,
			'mediaFile' => $filePath
		);
				
		return $this->responseView('XenGallery_ViewPublic_Media_Download', '', $viewParams);
	}
	
	/**
	* Displays a form to like a picture.
	*
	* @return XenForo_ControllerResponse_Abstract
	*/
	public function actionLike()
	{
		$visitor = XenForo_Visitor::getInstance();
		
		$mediaModel = $this->_getMediaModel();
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
		
		if (!$mediaModel->canLikeMedia($media, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
		
		$likeModel = $this->_getLikeModel();

		$existingLike = $likeModel->getContentLikeByLikeUser('xengallery_media', $mediaId, XenForo_Visitor::getUserId());
		
		if ($this->_request->isPost())
		{
			if ($existingLike)
			{
				$latestUsers = $likeModel->unlikeContent($existingLike);
			}
			else
			{
				$latestUsers = $likeModel->likeContent('xengallery_media', $mediaId, $media['user_id']);
			}

			$liked = ($existingLike ? false : true);
			
			if ($this->_noRedirect() && $latestUsers !== false)
			{				
				$media['likeUsers'] = $latestUsers;
				$media['likes'] += ($liked ? 1 : -1);
				$media['like_date'] = ($liked ? XenForo_Application::$time : 0);

				$viewParams = array(
					'media' => $media,
					'liked' => $liked
				);

				return $this->responseView('XenGallery_ViewPublic_Media_LikeConfirmed', '', $viewParams);
			}
		}
		else
		{
			throw $this->getNoPermissionResponseException();
		}
	}
	
	public function actionReport()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		
		$mediaHelper = $this->_getMediaHelper();
		$media = $mediaHelper->assertMediaValidAndViewable($mediaId);
		
		if ($this->isConfirmedPost())
		{
			$reportMessage = $this->_input->filterSingle('message', XenForo_Input::STRING);
			if (!$reportMessage)
			{
				return $this->responseError(new XenForo_Phrase('xengallery_please_enter_reason_for_reporting_this_media'));
			}

			$this->assertNotFlooding('report');
						
			/* @var $reportModel XenForo_Model_Report */
			$reportModel = XenForo_Model::create('XenForo_Model_Report');
			$reportModel->reportContent('xengallery_media', $media, $reportMessage);

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $media),
				new XenForo_Phrase('xengallery_thank_you_for_reporting_this_media')
			);
		}
		else
		{						
			$viewParams = array(
				'media' => $media,
				'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media)
			);

			return $this->responseView('XenGallery_ViewPublic_Media_Report', 'xengallery_media_report', $viewParams);				
		}		
	}
	
	public function actionUser()
	{
		$userId = $this->_input->filterSingle('user_id', XenForo_Input::UINT);
		
		if ($userId)
		{
			return $this->responseReroute(__CLASS__, 'userContent');
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery')
		);
	}
	
	public function actionUserContent()
	{
		$userId = $this->_input->filterSingle('user_id', XenForo_Input::UINT);
		$userFetchOptions = array(
			'join' => XenForo_Model_User::FETCH_LAST_ACTIVITY
		);
		$user = $this->getHelper('UserProfile')->assertUserProfileValidAndViewable($userId, $userFetchOptions);
		
		$mediaModel = $this->_getMediaModel();
		
		$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
		$perPage = XenForo_Application::get('options')->xengalleryMediaMaxPerPageUser;		
		
		$conditions = array(
			'user_id' => $user['user_id'],
			'deleted' => XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'xengallery', 'viewDeleted')
		);

		$fetchOptions = $this->_getMediaFetchOptions() + array(
			'order' => 'media_date',
			'orderDirection' => 'desc',
			'page' => $page,
			'perPage' => $perPage
		);
		
		$totalCount = $mediaModel->countMedia($conditions);		
		
		$media = $mediaModel->getMedia($conditions, $fetchOptions);
		$media = $mediaModel->prepareMediaItems($media);	
		
		$viewParams = array(
			'media' => $media,
			'user' => $user,
			'mediaCount' => count($media),
			'totalCount' => $totalCount,
			'page' => $page,
			'perPage' => $perPage,			
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_User', 'xengallery_media_user', $viewParams);
	}
	
	public function actionAddComment()
	{
		$this->_assertPostOnly();

		$this->_assertRegistrationRequired();
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaHelper->assertCanAddComment();		

		$this->assertNotFlooding('actionAddComment');

		$visitor = XenForo_Visitor::getInstance();
		$message = $this->_input->filterSingle('media_comment', XenForo_Input::STRING);

		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);

		$input = $this->_input->filter(array(
			'media_id' => XenForo_Input::UINT
		));

		$media = $mediaHelper->assertMediaValidAndViewable($input['media_id']);

		$input['media_comment'] = $this->getHelper('Editor')->getMessageText('media_comment', $this->_input);
		$input['media_comment'] = XenForo_Helper_String::autoLinkBbCode($input['media_comment']);

		$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');

		if ($commentId)
		{
			$writer->setExistingData($commentId);
			$writer->set('user_id', $writer->get('user_id'));
			$writer->set('media_comment', $input['media_comment']);
		}
		else 
		{
			$writer->set('media_id', $input['media_id']);
			$writer->set('user_id', $visitor->user_id);
			$writer->set('username', $visitor->username);
			$writer->set('media_comment', $input['media_comment']);
			// set in datawriter $writer->set('comment_date', XenForo_Application::$time);
		}

		$writer->save();

		// only run this code if the action has been loaded via XenForo.ajax()
		if ($this->_noRedirect())
		{
			$commentModel = $this->_getCommentModel();

			$date = $this->_input->filterSingle('date', XenForo_Input::UINT);

			if ($commentId)
			{
				$comment = $commentModel->getCommentById($commentId);

				$date = $comment['comment_date'] - 1;
			}

 			$comments = $commentModel->getCommentsNewerThan($date, $input['media_id']);
			$comments = $commentModel->prepareComments($comments); 			

			$date = $commentModel->getLatestDate();

			$viewParams = array(
				'comments' => $comments,
				'date' => $date,
				'media' => $media
			);

			return $this->responseView(
				'XenGallery_ViewPublic_Media_LatestComments',
				'xengallery_media_comments',
				$viewParams
			);
		}

		// redirect back to the normal scratchpad index page
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery', $media)
		);
	}	
	
	public function actionCategory()
	{
		$order = $this->_input->filterSingle('order', XenForo_Input::STRING);
		
		$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
		if (!$categoryId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$mediaHelper = $this->_getMediaHelper();
		$categoryModel = $this->_getCategoryModel();
		
		$category = $mediaHelper->assertCategoryValidAndViewable($categoryId);
		$categoryBreadcrumbs = $categoryModel->getCategoryBreadcrumb($category, false);
		
		$containerCategory = false;
		
		$uploadUserGroups = unserialize($category['upload_user_groups']);
		if (!$uploadUserGroups)
		{
			$canAddMedia = false;
			$containerCategory = true;
		}
		else
		{
			$canAddMedia = $this->_getMediaModel()->canAddMediaToCategory($uploadUserGroups);
		}
		
		$page = $this->_input->filterSingle('page', XenForo_Input::UINT);
		$perPage = XenForo_Application::get('options')->xengalleryMediaMaxPerPageCategory;
		
		$childIds = array();
		$showCategory = false;
		if ($containerCategory)
		{
			$childCategories = $categoryModel->getCategoryStructure(null, $categoryId);
			
			foreach ($childCategories AS $child)
			{
				$childIds[] = $child['xengallery_category_id'];
			}
			
			if (!$childCategories)
			{
				$containerCategory = false;
			}
			
			$showCategory = true;
		}
		
		$conditions = array(
			'xengallery_category_id' => $containerCategory ? $childIds : $categoryId,
			'deleted' => XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'xengallery', 'viewDeleted')
		);
		
		$fetchOptions = $this->_getMediaFetchOptions() + array(
			'order' => $order ? $order : 'media_date',
			'page' => $page,
			'perPage' => $perPage
		);
		
		$mediaModel = $this->_getMediaModel();
		
		$media = $mediaModel->getMedia($conditions, $fetchOptions);
		$media = $mediaModel->prepareMediaItems($media);
		
		$ignoredNames = array();
		foreach ($media AS $item)
		{
			if (isset($item['isIgnored']))
			{
				$ignoredNames[] = $item['username'];
			}
		}
		
		$mediaCount = $mediaModel->countMedia($conditions);
		
		$this->canonicalizePageNumber($page, $perPage, $mediaCount, 'xenmediagallery/categories', $category);
		$this->canonicalizeRequestUrl(
			XenForo_Link::buildPublicLink('xenmediagallery/categories', $category, array('page' => $page))
		);		
		
		$viewParams = array(
			'category' => $category,
			'containerCategory' => $containerCategory,
			'showCategory' => $showCategory,
			'media' => $media,
			'order' => $order,
			'ignoredNames' => array_unique($ignoredNames),
			'mediaCount' => $mediaCount,
			'page' => $page,
			'perPage' => $perPage,
			'canAddMedia' => $canAddMedia,
			'categoryBreadcrumbs' => $categoryBreadcrumbs
		);
		
		return $this->_getWrapper($categoryId,
			$this->responseView('XenGallery_ViewPublic_Media_HomeCategory', 'xengallery_category_view', $viewParams)
		);		
	}
	
	/**
	 * Gets the media add/edit form response.
	 *
	 * @param array $media
	 *
	 * @return XenForo_ControllerResponse_Abstract
	 */
	protected function _getMediaAddEditResponse($category, array $media = null)
	{
		if (!is_array($category))
		{
			$category = $this->_getMediaHelper()->assertCategoryValidAndViewable($category);
		}
		
		$mediaModel = $this->_getMediaModel();
		$categoryModel = $this->_getCategoryModel();
		
		$categories = $categoryModel->getCategoryStructure();
		$categories = $categoryModel->prepareCategories($categories);
		
		$attachmentParams = $mediaModel->getAttachmentParams($category);
		$attachmentConstraints = $mediaModel->getAttachmentConstraints();
		
		$allowedTypes = unserialize($category['allowed_types']);
        
        $mediaSites = $this->getModelFromCache('XenForo_Model_BbCode')->getAllBbCodeMediaSites();
        $allowedSites = XenForo_Application::get('options')->xengalleryMediaSites;
        
        foreach ($mediaSites AS $key => $mediaSite)
        {
            if (!in_array($mediaSite['media_site_id'], $allowedSites))
            {
                unset ($mediaSites[$key]);
            }
        }
		
		$viewParams = array(
			'category' => $category,
			'allowedTypes' => $allowedTypes,
			'categories' => $categories,
			'media' => $media,
			'embedUrl' => isset($media['media_embed_url']) ? $media['media_embed_url'] : false,
			'attachmentParams' => $attachmentParams,
			'attachmentConstraints' => $attachmentConstraints,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($category),
			'mediaSites' => $mediaSites
		);
		
		return $this->_getWrapper($category['xengallery_category_id'],
			$this->responseView('XenGallery_ViewPublic_Media_Edit', 'xengallery_media_edit', $viewParams)
		);			
	}
	
	public function actionAdd()
	{
		$this->_getMediaHelper()->assertCanAddMedia();
		
		if ($this->isConfirmedPost())
		{
			$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
			$category = $this->_getMediaHelper()->assertCategoryValidAndViewable($categoryId);
			
			if (!$this->_getMediaModel()->canAddMediaToCategory(unserialize($category['upload_user_groups']), $error))
			{
				throw $this->getErrorOrNoPermissionResponseException($error);
			}
			
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery/categories/add', $category),
				''
			);
		}
		else
		{
			$categoryModel = $this->_getCategoryModel();
			
			$categories = $categoryModel->getCategoryStructure();
			$categories = $categoryModel->prepareCategories($categories);
			
			$viewParams = array(
				'categories' => $categories
			);
			
			return $this->_getWrapper('',
				$this->responseView('XenForo_ViewPublic_Media_Add', 'xengallery_media_edit_pick_category', $viewParams)
			);			
		}		
	}
	
	public function actionEdit()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		if (!$mediaId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaModel = $this->_getMediaModel();
		
        $media = $mediaHelper->assertMediaValidAndViewable($mediaId);        
        $mediaHelper->assertCanEditMedia($media);
		$media = $mediaModel->prepareMedia($media);
		
		return $this->_getMediaAddEditResponse($media, $media);
	}
	
	public function actionDelete()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		if (!$mediaId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$hardDelete = $this->_input->filterSingle('hard_delete', XenForo_Input::UINT);
		$deleteType = ($hardDelete ? 'hard' : 'soft');		
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaModel = $this->_getMediaModel();
		
		$media = $mediaModel->getMediaById($mediaId, $this->_getMediaFetchOptions());	
		$media = $mediaModel->prepareMedia($media);
		$mediaHelper->assertCanDeleteMedia($media, $deleteType);
		
		if ($this->isConfirmedPost())
		{
			$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			$writer->setExistingData($media['media_id']);

			if ($hardDelete)
			{
				$writer->delete();
				
				// TODO: Implement Moderator Log Handling
			}
			else
			{
				$reason = $this->_input->filterSingle('reason', XenForo_Input::STRING);
				
				$writer->setExtraData(XenGallery_DataWriter_Media::DATA_DELETE_REASON, $reason);
				$writer->set('media_state', 'deleted');
				$writer->save();

				// TODO: Implement Moderator Log Handling
			}

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery/categories', $media)
			);
		}
		else
		{
			$viewParams = array(
				'media' => $media,
				'canHardDelete' => $mediaModel->canDeleteMedia($media, 'hard'),
				'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media)
			);
			
			return $this->responseView('XenGallery_ViewPublic_Media_Delete', 'xengallery_media_delete', $viewParams);			
		}	
	}
	
	public function actionUndelete()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		if (!$mediaId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaModel = $this->_getMediaModel();
		
		$media = $mediaModel->getMediaById($mediaId, $this->_getMediaFetchOptions());
		
		if ($media['media_state'] != 'deleted')
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery', $media)
			);
		}		

		$mediaHelper->assertCanDeleteMedia($media);
		
		if ($this->isConfirmedPost())
		{
			$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			$writer->setExistingData($media['media_id']);

			$writer->set('media_state', 'visible');
			$writer->save();

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $media)
			);
		}
		else
		{
			$viewParams = array(
				'media' => $media
			);
			
			return $this->responseView('XenGallery_ViewPublic_Media_Undelete', 'xengallery_media_undelete', $viewParams);
		}		
	}
	
	public function actionCaptionAdd()
	{	
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
		
		return $this->_getCaptionAddEditResponse($media);
	}	
	
	public function actionCaptionEdit()
	{	
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
				
		return $this->_getCaptionAddEditResponse($media);
	}
	
	protected function _getCaptionAddEditResponse(array $media, $inline = false)
	{	
		$this->_getMediaHelper()->assertCanEditCaption($media);
		
		$inline = $this->_input->filterSingle('inline', XenForo_Input::STRING);
		
		$viewParams = array(
			'media' => $media,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media, true)
		);
		
		if ($inline)
		{
			return $this->responseView('XenGallery_ViewPublic_Media_CaptionEditInline', 'xengallery_caption_edit_inline', $viewParams);
		}
		
		return $this->responseView('XenGallery_ViewPublic_Media_CaptionEdit', 'xengallery_caption_edit', $viewParams);
	}
	
	public function actionCaptionSave()
	{
		$mediaCaption = $this->getHelper('Editor')->getMessageText('media_caption', $this->_input);
		$mediaCaption = XenForo_Helper_String::autoLinkBbCode($mediaCaption);
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);
		
		$this->_getMediaHelper()->assertCanEditCaption($media);
		
		$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
		$writer->setExistingData($mediaId);
		$writer->set('media_caption', $mediaCaption);
		
		$writer->save();
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery', $media)
		);
	}
	
	public function actionIp()
	{
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);

		if (!XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'general', 'viewIps'))
		{
			throw $this->getErrorOrNoPermissionResponseException();
		}

		if (!$media['ip_id'])
		{
			return $this->responseError(new XenForo_Phrase('no_ip_information_available'));
		}

		$viewParams = array(
			'media' => $media,
			'categoryBreadCrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($media),
			'ipInfo' => $this->getModelFromCache('XenForo_Model_Ip')->getContentIpInfo($media)
		);

		return $this->responseView('XenGallery_ViewPublic_Media_Ip', 'xengallery_media_ip', $viewParams);
	}		
	
	public function actionApprove()
	{
		$this->_checkCsrfFromToken($this->_input->filterSingle('t', XenForo_Input::STRING));
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);

		if (!$this->_getMediaModel()->canApproveMedia($media, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}

		$dw = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
		$dw->setExistingData($media['media_id']);
		$dw->set('media_state', 'visible');
		$dw->save();

		//XenForo_Model_Log::logModeratorAction('xenfoto', $media, 'unapprove');

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery', $media)
		);
	}

	public function actionUnapprove()
	{
		$this->_checkCsrfFromToken($this->_input->filterSingle('t', XenForo_Input::STRING));
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		
		$media = $this->_getMediaHelper()->assertMediaValidAndViewable($mediaId);

		if (!$this->_getMediaModel()->canUnapproveMedia($media, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}

		$dw = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
		$dw->setExistingData($media['media_id']);
		$dw->set('media_state', 'moderated');
		$dw->save();

		//XenForo_Model_Log::logModeratorAction('xenfoto', $media, 'unapprove');

		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery', $media)
		);
	}
	
	public function actionPreviewVideo()
	{
		$embedUrl = $this->_input->filterSingle('embed_url', XenForo_Input::STRING);
		$mediaTag = XenForo_Helper_Media::convertMediaLinkToEmbedHtml($embedUrl);
        
        if (!$mediaTag)
        {
            return $this->responseView('XenGallery_ViewPublic_Media_PreviewVideo', '', array('throwError' => true, 'notValid' => true));
        }        
        
        preg_match('#\[media=(.*)\](.+)\[\/media\]#i', $mediaTag, $matches);        
        
        $allowedSites = XenForo_Application::get('options')->xengalleryMediaSites;
        
        if (!in_array($matches[1], $allowedSites))
        {   
            return $this->responseView('XenGallery_ViewPublic_Media_PreviewVideo', '', array('throwError' => true, 'notAllowed' => true));
        }
		
		$mediaId = $this->_input->filterSingle('media_id', XenForo_Input::UINT);
		$media = $this->_getMediaModel()->getMediaById($mediaId);
		
		$viewParams = array(
			'mediaTag' => $mediaTag,
			'uniqueId' => uniqid(),
			'embedUrl' => $embedUrl,
			'item' => $media,
			'isEditable' => true
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_PreviewVideo', 'xengallery_media_thumb', $viewParams);
	}
	
	public function actionCategoryAdd()
	{		
		$categoryId = $this->_input->filterSingle('xengallery_category_id', XenForo_Input::UINT);
		if (!$categoryId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$this->_getMediaHelper()->assertCanAddMedia();
		
		$category = $this->_getMediaHelper()->assertCategoryValidAndViewable($categoryId);
		
		if (!$this->_getMediaModel()->canAddMediaToCategory(unserialize($category['upload_user_groups']), $error))
		{
			throw $this->getErrorOrNoPermissionResponseException($error);
		}		
		
		return $this->_getMediaAddEditResponse($category, array('xengallery_category_id' => $categoryId));
	}
	
	public function actionCategorySaveMedia()
	{
		$this->_assertPostOnly();
		
		$input = $this->_input->filter(array(
			'media_id' => XenForo_Input::UINT,
			'xengallery_category_id' => XenForo_Input::UINT,
			'new_xengallery_category_id' => XenForo_Input::UINT,
			'attachment_hash' => XenForo_Input::STRING,
			'media_title' => XenForo_Input::ARRAY_SIMPLE,
			'media_description' => XenForo_Input::ARRAY_SIMPLE,
			'media_tag' => XenForo_Input::ARRAY_SIMPLE,
			'media_embed_url' => XenForo_Input::ARRAY_SIMPLE
		));
        
        $button = $this->_input->filter(array(
            'save_embed' => XenForo_Input::STRING,
            'save_upload' => XenForo_Input::STRING,        
        ));
		
		if ($input['media_id'])
		{
			$mediaModel = $this->_getMediaModel();
			
			$media = $mediaModel->getMediaById($input['media_id']);
			$this->_getMediaHelper()->assertCanEditMedia($media);
		}
		
		if (empty($input['media_title']))
		{
			return $this->responseError(new XenForo_Phrase('xengallery_you_have_not_added_any_media'));
		}
		
		$visitor = XenForo_Visitor::getInstance();

		$mediaState = 'moderated';
		if (XenForo_Permission::hasPermission($visitor['permissions'], 'xengallery', 'followModerationRules'))
		{
			$mediaState = 'visible';
		}
		
		$input['media_state'] = $mediaState;
		
		$category = $this->_getMediaHelper()->assertCategoryValidAndViewable($input['xengallery_category_id']);
		                                                                                                       
		if ($input['media_id'])
		{
			unset ($input['attachment_hash']);
			                                                       
			$mediaInput = array(
				'media_title' => array_shift($input['media_title']),
				'media_description' => array_shift($input['media_description']),
				'media_tag' => array_shift($input['media_tag']),
				'media_embed_url' => array_shift($input['media_embed_url'])
			);
			
			$input = array_merge($input, $mediaInput);
			
			if ($media['media_type'] == 'image_upload')
			{
				unset ($input['media_tag']);
				unset ($input['media_embed_url']);
			}			
			
			if ($input['new_xengallery_category_id'])
			{
				$input['xengallery_category_id'] = $input['new_xengallery_category_id'];
				unset ($input['new_xengallery_category_id']);
			}
			
			$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Media');
			
			$writer->setExistingData($input['media_id']);
			$writer->bulkSet($input);
			
			$writer->save();
            
            return $this->responseRedirect(
                XenForo_ControllerResponse_Redirect::SUCCESS,
                $input['media_id'] ? XenForo_Link::buildPublicLink('xenmediagallery', $input) : XenForo_Link::buildPublicLink('xenmediagallery/categories', $category),
                new XenForo_Phrase('xengallery_media_has_been_saved')
            );
		}
		
        if ($button['save_embed'])
		{
			$visitor = XenForo_Visitor::getInstance();

			$i = 0;
			foreach ($input['media_title'] AS $key => $video)
			{
				$i++;
				$mediaWriter = XenForo_DataWriter::create('XenGallery_DataWriter_Media');				
				
				$mediaData = array();
				$mediaData['media_title'] = $input['media_title'][$key];
				$mediaData['media_description'] = $input['media_description'][$key];
				$mediaData['media_tag'] = $input['media_tag'][$key];
				$mediaData['media_embed_url'] = $input['media_embed_url'][$key];
				
				$mediaData['media_type'] = 'video_embed';
				
				$mediaData += array(
					'media_state' => $input['media_state'],
					'media_date' => XenForo_Application::$time + $i,
					'xengallery_category_id' => $category['xengallery_category_id'],
					'user_id' => $visitor->user_id,
					'username' => $visitor->username
				);
				
				$mediaWriter->bulkSet($mediaData);
				$mediaWriter->save();				
			}
		}
		elseif ($button['save_upload'])
		{
			$attachmentModel = $this->_getAttachmentModel();
			$mediaModel = $this->_getMediaModel();
			
			$attachments = $attachmentModel->getAttachmentsByTempHash($input['attachment_hash']);
			$attachments = $mediaModel->prepareMediaItems($attachments);
			
			$mediaModel->associateAttachmentsAndMedia($attachments, $input, $category);
		}
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			$input['media_id'] ? XenForo_Link::buildPublicLink('xenmediagallery', $input) : XenForo_Link::buildPublicLink('xenmediagallery/categories', $category),
			new XenForo_Phrase('xengallery_media_has_been_saved')
		);		
	}
	
	public function actionFileUpload()
	{
		$input = $this->_input->filter(array(
			'hash' => XenForo_Input::STRING,
			'content_type' => XenForo_Input::STRING,
			'content_data' => array(XenForo_Input::UINT, 'array' => true),
			'key' => XenForo_Input::STRING
		));
		if (!$input['hash'])
		{
			$input['hash'] = $this->_input->filterSingle('temp_hash', XenForo_Input::STRING);
		}

		$this->_assertCanUploadAndManageAttachments($input['hash'], $input['content_type'], $input['content_data']);

		$attachmentModel = $this->_getAttachmentModel();
		$attachmentHandler = $attachmentModel->getAttachmentHandler($input['content_type']); // known to be valid
		$contentId = $attachmentHandler->getContentIdFromContentData($input['content_data']);

		$existingAttachments = ($contentId
			? $attachmentModel->getAttachmentsByContentId($input['content_type'], $contentId)
			: array()
		);
		$newAttachments = $attachmentModel->getAttachmentsByTempHash($input['hash']);

		$constraints = $attachmentHandler->getAttachmentConstraints();
		if ($constraints['count'] <= 0)
		{
			$canUpload = true;
			$remainingUploads = true;
		}
		else
		{
			$remainingUploads = $constraints['count'] - (count($existingAttachments) + count($newAttachments));
			$canUpload = ($remainingUploads > 0);
		}

		$viewParams = array(
			'attachmentConstraints' => $constraints,
			'existingAttachments' => $existingAttachments,
			'newAttachments' => $newAttachments,

			'canUpload' => $canUpload,
			'remainingUploads' => $remainingUploads,

			'hash' => $input['hash'],
			'contentType' => $input['content_type'],
			'contentData' => $input['content_data'],
			'attachmentParams' => array(
				'hash' => $input['hash'],
				'content_type' => $input['content_type'],
				'content_data' => $input['content_data']
			),
			'key' => $input['key']
		);

		return $this->responseView('XenGallery_ViewPublic_Media_Upload', 'xengallery_media_file_upload', $viewParams);
	}
	
	public function actionFileDoUpload()
	{
		$this->_assertPostOnly();

		$deleteArray = array_keys($this->_input->filterSingle('delete', XenForo_Input::ARRAY_SIMPLE));
		$delete = reset($deleteArray);
		if ($delete)
		{
			$this->_request->setParam('attachment_id', $delete);
			return $this->responseReroute(__CLASS__, 'delete');
		}

		$input = $this->_input->filter(array(
			'hash' => XenForo_Input::STRING,
			'content_type' => XenForo_Input::STRING,
			'content_data' => array(XenForo_Input::UINT, 'array' => true),
			'key' => XenForo_Input::STRING
		));
		if (!$input['hash'])
		{
			$input['hash'] = $this->_input->filterSingle('temp_hash', XenForo_Input::STRING);
		}
		
		$this->_assertCanUploadAndManageAttachments($input['hash'], $input['content_type'], $input['content_data']);		

		$attachmentModel = $this->_getAttachmentModel();
		$attachmentHandler = $attachmentModel->getAttachmentHandler($input['content_type']); // known to be valid
		$contentId = $attachmentHandler->getContentIdFromContentData($input['content_data']);

		$existingAttachments = ($contentId
			? $attachmentModel->getAttachmentsByContentId($input['content_type'], $contentId)
			: array()
		);
		$newAttachments = $attachmentModel->getAttachmentsByTempHash($input['hash']);

		$attachmentConstraints = $attachmentHandler->getAttachmentConstraints();

		if ($attachmentConstraints['count'] > 0)
		{
			$remainingUploads = $attachmentConstraints['count'] - (count($existingAttachments) + count($newAttachments));
			if ($remainingUploads <= 0)
			{
				return $this->responseError(new XenForo_Phrase(
					'you_may_not_upload_more_files_with_message_allowed_x',
					array('total' => $attachmentConstraints['count'])
				));
			}
		}

		$file = XenForo_Upload::getUploadedFile('upload');
		if (!$file)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery/files/upload', false, array(
					'hash' => $input['hash'],
					'content_type' => $input['content_type'],
					'content_data' => $input['content_data'],
					'key' => $input['key']
				))
			);
		}

		$file->setConstraints($attachmentConstraints);
		if (!$file->isValid())
		{
			return $this->responseError($file->getErrors());
		}
		
		$fileModel = $this->_getFileModel();
		
		$dataId = $fileModel->insertUploadedAttachmentData($file, XenForo_Visitor::getUserId());
		$attachmentId = $attachmentModel->insertTemporaryAttachment($dataId, $input['hash']);

		$message = new XenForo_Phrase('upload_completed_successfully');

		// return a view if noredirect has been requested and we are not deleting
		if ($this->_noRedirect())
		{
			$contentId = $attachmentHandler->getContentIdFromContentData($input['content_data']);

			//$newAttachments = $attachmentModel->getAttachmentsByTempHash($input['hash']);

			$attachment = $attachmentModel->getAttachmentById($attachmentId);

			$viewParams = array(
				'attachment' => $this->_getMediaModel()->prepareMedia($attachment),
				'message' => $message,
				'hash' => $input['hash'],
				'content_type' => $input['content_type'],
				'content_data' => $input['content_data'],
				'key' => $input['key']
			);

			return $this->responseView('XenGallery_ViewPublic_Media_DoUpload', '', $viewParams);
		}
		else
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery/files/upload', false, array(
					'hash' => $input['hash'],
					'content_type' => $input['content_type'],
					'content_data' => $input['content_data'],
					'key' => $input['key']
				)),
				$message
			);
		}		
	}
	
	public function actionCommentEdit()
	{
		$commentModel = $this->_getCommentModel();
		$categoryModel = $this->_getCategoryModel();
		
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		
		$fetchOptions = array(
			'join' =>	XenGallery_Model_Comment::FETCH_USER | XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_CATEGORY
		);
		
		$comment = $commentModel->getCommentById($commentId, $fetchOptions);
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaHelper->assertCanEditComment($comment);
		
		$viewParams = array(
			'comment' => $comment,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment),
			'commentEditor' => true
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_CommentEdit', 'xengallery_comment_edit', $viewParams);
	}
	
	public function actionCommentEditInline()
	{
		$commentModel = $this->_getCommentModel();
		$categoryModel = $this->_getCategoryModel();
		
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		
		$fetchOptions = array(
			'join' =>	XenGallery_Model_Comment::FETCH_USER | XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_CATEGORY
		);
		
		$comment = $commentModel->getCommentById($commentId, $fetchOptions);
		
		$mediaHelper = $this->_getMediaHelper();
		$mediaHelper->assertCanEditComment($comment);
		
		$viewParams = array(
			'comment' => $comment,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment, true)
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_CommentEditInline', 'xengallery_comment_edit_inline', $viewParams);
	}
	
	public function actionCommentDelete()
	{
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		if (!$commentId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$hardDelete = $this->_input->filterSingle('hard_delete', XenForo_Input::UINT);
		$deleteType = ($hardDelete ? 'hard' : 'soft');		
		
		$mediaHelper = $this->_getMediaHelper();
		$commentModel = $this->_getCommentModel();
		
		$fetchOptions = array(
			'join' =>	XenGallery_Model_Comment::FETCH_USER | XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_CATEGORY
		);
		
		$comment = $commentModel->getCommentById($commentId, $fetchOptions);
		
		$mediaHelper->assertCanDeleteComment($comment, $deleteType);
		
		if ($this->isConfirmedPost())
		{
			$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');
			$writer->setExistingData($comment['comment_id']);

			if ($hardDelete)
			{
				$writer->delete();
				
				// TODO: Implement Moderator Log Handling
			}
			else
			{
				$reason = $this->_input->filterSingle('reason', XenForo_Input::STRING);
				
				$writer->setExtraData(XenGallery_DataWriter_Comment::DATA_DELETE_REASON, $reason);
				$writer->set('comment_state', 'deleted');
				$writer->save();

				// TODO: Implement Moderator Log Handling
			}

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $comment)
			);
		}
		else
		{					
			$viewParams = array(
				'comment' => $comment,
				'canHardDelete' => $commentModel->canDeleteComment($comment, 'hard'),
				'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment)
			);
			
			return $this->responseView('XenGallery_ViewPublic_Media_CommentDelete', 'xengallery_comment_delete', $viewParams);			
		}	
	}
	
	public function actionCommentUndelete()
	{
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		if (!$commentId)
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery')
			);
		}
		
		$mediaHelper = $this->_getMediaHelper();
		$commentModel = $this->_getCommentModel();
		
		$comment = $commentModel->getCommentById($commentId, $this->_getCommentFetchOptions());
		
		if ($comment['comment_state'] != 'deleted')
		{
			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::RESOURCE_CANONICAL_PERMANENT,
				XenForo_Link::buildPublicLink('xenmediagallery', $comment)
			);
		}		

		$mediaHelper->assertCanDeleteComment($comment);
		
		if ($this->isConfirmedPost())
		{
			$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');
			$writer->setExistingData($comment['comment_id']);

			$writer->set('comment_state', 'visible');
			$writer->save();

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $comment)
			);
		}
		else
		{
			$viewParams = array(
				'comment' => $comment
			);
			
			return $this->responseView('XenGallery_ViewPublic_Media_CommentUndelete', 'xengallery_comment_undelete', $viewParams);
		}		
	}
	
	public function actionCommentIp()
	{
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		$comment = $this->_getMediaHelper()->assertCommentValidAndViewable($commentId);

		if (!XenForo_Permission::hasPermission(XenForo_Visitor::getInstance()->permissions, 'general', 'viewIps'))
		{
			throw $this->getErrorOrNoPermissionResponseException();
		}

		if (!$comment['ip_id'])
		{
			return $this->responseError(new XenForo_Phrase('no_ip_information_available'));
		}

		$viewParams = array(
			'comment' => $comment,
			'categoryBreadCrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment),
			'ipInfo' => $this->getModelFromCache('XenForo_Model_Ip')->getContentIpInfo($comment)
		);

		return $this->responseView('XenGallery_ViewPublic_Media_CommentIp', 'xengallery_comment_ip', $viewParams);
	}	
	
	public function actionCommentSave()
	{
		$this->_assertPostOnly();
		$this->_assertRegistrationRequired();		
		
		$commentModel = $this->_getCommentModel();
		$categoryModel = $this->_getCategoryModel();
		
		$message = $this->_input->filterSingle('media_comment', XenForo_Input::STRING);
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		
		$fetchOptions = array(
			'join' =>	XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_RATING
		);
		
		$comment = $commentModel->getCommentById($commentId, $fetchOptions);
		
		$input['media_comment'] = $this->getHelper('Editor')->getMessageText('media_comment', $this->_input);
		$input['media_comment'] = XenForo_Helper_String::autoLinkBbCode($input['media_comment']);

		$writer = XenForo_DataWriter::create('XenGallery_DataWriter_Comment');
		
		$writer->setExistingData($commentId);
		$writer->set('media_comment', $input['media_comment']);
		
		$writer->save();
		
		// only run this code if the action has been loaded via XenForo.ajax()
		if ($this->_noRedirect())
		{
			if ($commentId)
			{
				$fetchOptions['join'] |= XenGallery_Model_Comment::FETCH_USER;
				$comment = $commentModel->getCommentById($commentId, $fetchOptions);
				$comment = $commentModel->prepareComments($comment);
			}

			$viewParams = array(
				'comment' => $comment
			);

			return $this->responseView(
				'XenGallery_ViewPublic_Media_Save_CommentListItem',
				'',
				$viewParams
			);
		}		
		
		return $this->responseRedirect(
			XenForo_ControllerResponse_Redirect::SUCCESS,
			XenForo_Link::buildPublicLink('xenmediagallery', $comment) . "#comment-$commentId",
			new XenForo_Phrase('xengallery_comment_updated_successfully')
		);
	}
	
	public function actionCommentShow()
	{
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		$comment = $this->_getMediaHelper()->assertCommentValidAndViewable($commentId);
		$comment = $this->_getCommentModel()->prepareComments($comment);
		
		$viewParams = array(
			'comment' => $comment,
			'commentId' => $commentId
		);
		
		return $this->responseView(
			'XenGallery_ViewPublic_Media_ShowComment',
			'',
			$viewParams
		);		
	}
	
	public function actionCommentLike()
	{
		$visitor = XenForo_Visitor::getInstance();
		
		$commentModel = $this->_getCommentModel();
		
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		$comment = $commentModel->getCommentById($commentId, $this->_getCommentFetchOptions());
		
		if (!$commentModel->canLikeComment($comment, $errorPhraseKey))
		{
			throw $this->getErrorOrNoPermissionResponseException($errorPhraseKey);
		}
		
		$likeModel = $this->_getLikeModel();

		$existingLike = $likeModel->getContentLikeByLikeUser('xengallery_comment', $commentId, XenForo_Visitor::getUserId());
		
		if ($this->_request->isPost())
		{
			if ($existingLike)
			{
				$latestUsers = $likeModel->unlikeContent($existingLike);
			}
			else
			{
				$latestUsers = $likeModel->likeContent('xengallery_comment', $commentId, $comment['user_id']);
			}

			$liked = ($existingLike ? false : true);
			
			if ($this->_noRedirect() && $latestUsers !== false)
			{				
				$comment['likeUsers'] = $latestUsers;
				$comment['likes'] += ($liked ? 1 : -1);
				$comment['like_date'] = ($liked ? XenForo_Application::$time : 0);

				$viewParams = array(
					'comment' => $comment,
					'liked' => $liked
				);

				return $this->responseView('XenGallery_ViewPublic_Media_LikeConfirmedComment', '', $viewParams);
			}
		}
		else
		{
			throw $this->getNoPermissionResponseException();
		}
	}
	
	public function actionCommentLikes()
	{
		$mediaHelper = $this->_getMediaHelper();
		
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		$comment =  $mediaHelper->assertCommentValidAndViewable($commentId);
		
		$likes = $this->_getLikeModel()->getContentLikes('xengallery_comment', $commentId);
		if (!$likes)
		{
			return $this->responseError(new XenForo_Phrase('xengallery_no_one_has_liked_this_comment_yet'));
		}
					
		$viewParams = array(
			'comment' => $comment,
			'likes' => $likes,
			'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment)
		);
		
		return $this->responseView('XenGallery_ViewPublic_Media_Ratings', 'xengallery_comment_likes', $viewParams);		
	}	
	
	public function actionCommentReport()
	{
		$commentId = $this->_input->filterSingle('comment_id', XenForo_Input::UINT);
		
		$mediaHelper = $this->_getMediaHelper();
		$comment = $mediaHelper->assertCommentValidAndViewable($commentId);
		
		if ($this->isConfirmedPost())
		{
			$reportMessage = $this->_input->filterSingle('message', XenForo_Input::STRING);
			if (!$reportMessage)
			{
				return $this->responseError(new XenForo_Phrase('xengallery_please_enter_reason_for_reporting_this_media'));
			}

			$this->assertNotFlooding('report');
						
			/* @var $reportModel XenForo_Model_Report */
			$reportModel = XenForo_Model::create('XenForo_Model_Report');
			$reportModel->reportContent('xengallery_comment', $comment, $reportMessage);

			return $this->responseRedirect(
				XenForo_ControllerResponse_Redirect::SUCCESS,
				XenForo_Link::buildPublicLink('xenmediagallery', $comment),
				new XenForo_Phrase('xengallery_thank_you_for_reporting_this_comment')
			);
		}
		else
		{
			$viewParams = array(
				'comment' => $comment,
				'categoryBreadcrumbs' => $this->_getCategoryModel()->getCategoryBreadcrumb($comment)
			);

			return $this->responseView('XenGallery_ViewPublic_Media_CommentReport', 'xengallery_comment_report', $viewParams);				
		}		
	}
	
	public static function getSessionActivityDetailsForList(array $activities)
	{
		$mediaIds = array();
		$categoryIds = array();
		$userIds = array();
		foreach ($activities AS $activity)
		{
			if (!empty($activity['params']['media_id']))
			{
				$mediaIds[$activity['params']['media_id']] = intval($activity['params']['media_id']);
			}
			
			if (!empty($activity['params']['xengallery_category_id']))
			{
				$categoryIds[$activity['params']['xengallery_category_id']] = intval($activity['params']['xengallery_category_id']);
			}
			
			if (!empty($activity['params']['user_id']))
			{
				$userIds[$activity['params']['user_id']] = intval($activity['params']['user_id']);
			}
		}
		
		$mediaData = array();
		if ($mediaIds)
		{
			$mediaModel = XenForo_Model::create('XenGallery_Model_Media');

			$media = $mediaModel->getMedia(array('media_id' => $mediaIds), array());
			foreach ($media AS $item)
			{
				$mediaData[$item['media_id']] = array(
					'title' => $item['media_title'],
					'url' => XenForo_Link::buildPublicLink('xenmediagallery', $item)
				);
			}
		}
		
		$categoryData = array();
		if ($categoryIds)
		{
			$categoryModel = XenForo_Model::create('XenGallery_Model_Category');
			
			$categories = $categoryModel->getCategoriesByIds($categoryIds);
			foreach ($categories AS $category)
			{
				$categoryData[$category['xengallery_category_id']] = array(
					'title' => $category['category_title'],
					'url' => XenForo_Link::buildPublicLink('xenmediagallery/categories', $category)
				);
			}
		}
		
		$userData = array();
		if ($userIds)
		{
			$userModel = XenForo_Model::create('XenForo_Model_User');
			
			$users = $userModel->getUsersByIds($userIds);
			foreach ($users AS $user)
			{
				$userData[$user['user_id']] = array(
					'username' => $user['username'],
					'url' => XenForo_Link::buildPublicLink('xenmediagallery/users', $user)
				);
			}
		}
		
		$output = array();
		foreach ($activities AS $key => $activity)
		{
			$media = false;
			if (!empty($activity['params']['media_id']))
			{
				$mediaId = $activity['params']['media_id'];
				if (isset($mediaData[$mediaId]))
				{
					$media = $mediaData[$mediaId];
				}
			}

			if ($media)
			{
				$output[$key] = array(
					new XenForo_Phrase('xengallery_viewing_media'),
					$media['title'],
					$media['url'],
					''
				);
			}
			
			$category = false;
			if (!empty($activity['params']['xengallery_category_id']))
			{
				$categoryId = $activity['params']['xengallery_category_id'];
				if (isset($categoryData[$categoryId]))
				{
					$category = $categoryData[$categoryId];
				}
			}
			
			if ($category)
			{
				$output[$key] = array(
					new XenForo_Phrase('xengallery_viewing_media_category'),
					$category['title'],
					$category['url'],
					''
				);				
			}
			
			$user = false;
			if (!empty($activity['params']['user_id']))
			{
				$userId = $activity['params']['user_id'];
				if (isset($userData[$userId]))
				{
					$user = $userData[$userId];
				}
			}
			
			if ($user)
			{
				$output[$key] = array(
					new XenForo_Phrase('xengallery_viewing_media_user'),
					$user['username'],
					$user['url'],
					''
				);				
			}
		}
		
		if (!$output)
		{
			$output[$key] = new XenForo_Phrase('xengallery_viewing_media');
		}

		return $output;
	}	
	
	protected function _getWrapper($selected, XenForo_ControllerResponse_View $subView)
	{
		$mediaModel = $this->_getMediaModel();
		
		$categoryModel = $this->_getCategoryModel();
		$allCategories = $categoryModel->getCategoryStructure();
		
		$options = XenForo_Application::get('options');
		
		$comments = array();
		if ($options->xengalleryShowRecentComments)
		{
			$fetchOptions = array(
				'limit' => $options->xengalleryRecentCommentsLimit,
				'join' => XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_USER,
				'order' => 'comment_date',
				'orderDirection' => 'DESC'
			);
			
			$comments = $this->_getCommentModel()->getComments(array(), $fetchOptions);
			
			foreach ($comments AS $commentId => &$comment)
			{
				if (!$mediaModel->canViewMediaItem($comment))
				{
					unset ($comments[$commentId]);
				}
			}
		}
		
		$users = array();
		if ($options->xengalleryShowTopContributors)
		{
			$users = $mediaModel->getTopContributors($options->xengalleryTopContributorsLimit);
		}
		
		$viewParams = array(
			'selected' => $selected,
			'categories' => $allCategories,
			'recentComments' => $comments ? true : false,
			'comments' => $comments,
			'topContributors' => $users ? true : false,
			'users' => $users
		);

		$wrapper = $this->responseView('XenForo_ViewPublic_Media_Wrapper', 'xengallery_wrapper', $viewParams);
		$wrapper->subView = $subView;

		return $wrapper;
	}
	
	/**
	 * Asserts that the viewing user can upload and manage XenGallery files.
	 *
	 * @param string $hash Unique hash
	 * @param string $contentType
	 * @param array $contentData
	 */
	protected function _assertCanUploadAndManageAttachments($hash, $contentType, array $contentData)
	{
		if (!$hash)
		{
			throw $this->getNoPermissionResponseException();
		}

		$attachmentHandler = $this->_getAttachmentModel()->getAttachmentHandler($contentType);
		if (!$attachmentHandler || !$attachmentHandler->canUploadAndManageAttachments($contentData))
		{
			 throw $this->getNoPermissionResponseException();
		}
	}
	
	/**
	 * Gets the specified attachment or throws an error.
	 *
	 * @param integer $attachment
	 *
	 * @return array
	 */
	protected function _getAttachmentOrError($attachmentId)
	{
		$attachment = $this->_getAttachmentModel()->getAttachmentById($attachmentId);
		if (!$attachment)
		{
			throw $this->responseException($this->responseError(new XenForo_Phrase('requested_attachment_not_found'), 404));
		}

		return $attachment;
	}
	
	protected function _getMediaFetchOptions()
	{
		$mediaFetchOptions = array(
			'join' => XenGallery_Model_Media::FETCH_USER | XenGallery_Model_Media::FETCH_CATEGORY | XenGallery_Model_Media::FETCH_ATTACHMENT
		);
		
		if (XenForo_Visitor::getInstance()->hasPermission('xengallery', 'viewDeleted'))
		{
			$mediaFetchOptions['join'] |= XenGallery_Model_Media::FETCH_DELETION_LOG;
		}

		return $mediaFetchOptions;
	}
	
	protected function _getCommentFetchOptions()
	{
		$commentFetchOptions = array(
			'join' => XenGallery_Model_Comment::FETCH_USER | XenGallery_Model_Comment::FETCH_CATEGORY | XenGallery_Model_Comment::FETCH_MEDIA | XenGallery_Model_Comment::FETCH_RATING
		);
		
		if (XenForo_Visitor::getInstance()->hasPermission('xengallery', 'viewDeletedComments'))
		{
			$commentFetchOptions['join'] |= XenGallery_Model_Comment::FETCH_DELETION_LOG;
		}

		return $commentFetchOptions;
	}		
	
	/**
	 * @return XenGallery_ControllerHelper_Media
	 */
	protected function _getMediaHelper()
	{
		return $this->getHelper('XenGallery_ControllerHelper_Media');
	}	
	
	/**
	 * @return XenGallery_Model_Media
	 */	
	protected function _getMediaModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Media');
	}
	
	/**
	 * @return XenGallery_Model_Category
	 */
	protected function _getCategoryModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Category');
	}	
	
	/**
	 * @return XenForo_Model_Attachment
	 */
	protected function _getAttachmentModel()
	{
		return $this->getModelFromCache('XenForo_Model_Attachment');
	}
	
	/**
	* @return XenForo_Model_Like
	*/
	protected function _getLikeModel()
	{
		return $this->getModelFromCache('XenForo_Model_Like');
	}
	
	/**
	 * @return XenGallery_Model_Comment
	 */
	protected function _getCommentModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Comment');
	}	
	
	/**
	 * @return XenGallery_Model_Rating
	 */
	protected function _getRatingModel()
	{
		return $this->getModelFromCache('XenGallery_Model_Rating');
	}		
	
	/**
	 * @return XenForo_Model_User
	 */
	protected function _getUserModel()
	{
		return $this->getModelFromCache('XenForo_Model_User');
	}	
	
	/**
	 * @return XenGallery_Model_File
	 */
	protected function _getFileModel()
	{
		return $this->getModelFromCache('XenGallery_Model_File');
	}					
}